/*
 * Decompiled with CFR 0.152.
 */
package loci.formats.in;

import java.io.IOException;
import loci.common.DataTools;
import loci.common.RandomAccessInputStream;
import loci.formats.codec.JPEG2000BoxType;
import loci.formats.codec.JPEG2000SegmentMarker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JPEG2000MetadataParser {
    private static final Logger LOGGER = LoggerFactory.getLogger(JPEG2000MetadataParser.class);
    private RandomAccessInputStream in;
    private long codestreamOffset;
    private long maximumReadOffset;
    private Integer headerSizeX;
    private Integer headerSizeY;
    private Short headerSizeC;
    private Integer headerPixelType;
    private Integer codestreamSizeX;
    private Integer codestreamSizeY;
    private Short codestreamSizeC;
    private Integer codestreamPixelType;
    private boolean isRawCodestream = false;
    private Integer resolutionLevels;
    private int[][] lut;

    public JPEG2000MetadataParser(RandomAccessInputStream in) throws IOException {
        this.in = in;
        this.maximumReadOffset = in.length();
        this.parseBoxes();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public JPEG2000MetadataParser(RandomAccessInputStream in, long maximumReadOffset) throws IOException {
        this.in = in;
        this.maximumReadOffset = maximumReadOffset;
        boolean isLittleEndian = in.isLittleEndian();
        try {
            this.parseBoxes();
        }
        finally {
            in.order(isLittleEndian);
        }
    }

    public long getCodestreamOffset() {
        return this.codestreamOffset;
    }

    private void parseBoxes() throws IOException {
        long originalPos = this.in.getFilePointer();
        long nextPos = 0L;
        long pos = originalPos;
        LOGGER.trace("Parsing JPEG 2000 boxes at {}", pos);
        int length = 0;
        while (pos < this.maximumReadOffset) {
            pos = this.in.getFilePointer();
            length = this.in.readInt();
            int boxCode = this.in.readInt();
            JPEG2000BoxType boxType = JPEG2000BoxType.get(boxCode);
            if (boxType == JPEG2000BoxType.SIGNATURE_WRONG_ENDIANNESS) {
                LOGGER.trace("Swapping endianness during box parsing.");
                this.in.order(!this.in.isLittleEndian());
                length = DataTools.swap(length);
            }
            nextPos = pos + (long)length;
            if (length >= 8) {
                length -= 8;
            }
            if (boxType == null) {
                LOGGER.warn("Unknown JPEG 2000 box 0x{} at {}", (Object)Integer.toHexString(boxCode), (Object)pos);
                if (pos == originalPos) {
                    this.in.seek(originalPos);
                    if (JPEG2000SegmentMarker.get(this.in.readUnsignedShort()) != null) {
                        LOGGER.info("File is a raw codestream not a JP2.");
                        this.isRawCodestream = true;
                        this.in.seek(originalPos);
                        this.parseContiguousCodestream(this.in.length());
                    }
                }
            } else {
                LOGGER.trace("Found JPEG 2000 '{}' box at {}", (Object)boxType.getName(), (Object)pos);
                switch (boxType) {
                    case CONTIGUOUS_CODESTREAM: {
                        try {
                            this.parseContiguousCodestream(length == 0 ? this.in.length() : (long)length);
                        }
                        catch (Exception e) {
                            LOGGER.warn("Could not parse contiguous codestream.", e);
                        }
                        break;
                    }
                    case HEADER: {
                        this.in.skipBytes(4);
                        String s = this.in.readString(4);
                        if (s.equals("ihdr")) {
                            this.headerSizeY = this.in.readInt();
                            this.headerSizeX = this.in.readInt();
                            this.headerSizeC = this.in.readShort();
                            int type = this.in.readInt();
                            this.headerPixelType = this.convertPixelType(type);
                        }
                        this.parseBoxes();
                        break;
                    }
                    case PALETTE: {
                        int i;
                        short nEntries = this.in.readShort();
                        int nColumns = this.in.read();
                        int[] bitDepths = new int[nColumns];
                        for (i = 0; i < bitDepths.length; ++i) {
                            bitDepths[i] = this.in.read() & 0x7F;
                            while (bitDepths[i] % 8 != 0) {
                                int n = i;
                                bitDepths[n] = bitDepths[n] + 1;
                            }
                        }
                        this.lut = new int[nColumns][nEntries];
                        for (i = 0; i < nColumns; ++i) {
                            for (int j = 0; j < this.lut[i].length; ++j) {
                                if (bitDepths[i] == 8) {
                                    this.lut[i][j] = this.in.read();
                                    continue;
                                }
                                if (bitDepths[i] != 16) continue;
                                this.lut[i][j] = this.in.readShort();
                            }
                        }
                        break;
                    }
                }
            }
            if (nextPos < 0L || nextPos >= this.maximumReadOffset || length == 0) {
                LOGGER.trace("Exiting box parser loop.");
                break;
            }
            LOGGER.trace("Seeking to next box at {}", nextPos);
            this.in.seek(nextPos);
        }
    }

    private void parseContiguousCodestream(long length) throws IOException {
        if (this.codestreamOffset == 0L) {
            this.codestreamOffset = this.in.getFilePointer();
        }
        int segmentMarkerCode = 0;
        int segmentLength = 0;
        long pos = this.in.getFilePointer();
        long nextPos = 0L;
        LOGGER.trace("Parsing JPEG 2000 contiguous codestream of length {} at {}", length, (Object)pos);
        long maximumReadOffset = pos + length;
        boolean terminate = false;
        while (pos < maximumReadOffset && !terminate) {
            pos = this.in.getFilePointer();
            segmentMarkerCode = this.in.readUnsignedShort();
            JPEG2000SegmentMarker segmentMarker = JPEG2000SegmentMarker.get(segmentMarkerCode);
            if (segmentMarker == JPEG2000SegmentMarker.SOC_WRONG_ENDIANNESS) {
                LOGGER.trace("Swapping endianness during segment marker parsing.");
                this.in.order(!this.in.isLittleEndian());
                segmentMarkerCode = JPEG2000SegmentMarker.SOC.getCode();
                segmentMarker = JPEG2000SegmentMarker.SOC;
            }
            segmentLength = segmentMarker == JPEG2000SegmentMarker.SOC || segmentMarker == JPEG2000SegmentMarker.SOD || segmentMarker == JPEG2000SegmentMarker.EPH || segmentMarker == JPEG2000SegmentMarker.EOC || segmentMarkerCode >= JPEG2000SegmentMarker.RESERVED_DELIMITER_MARKER_MIN.getCode() && segmentMarkerCode <= JPEG2000SegmentMarker.RESERVED_DELIMITER_MARKER_MAX.getCode() ? 0 : this.in.readUnsignedShort();
            nextPos = pos + (long)segmentLength + 2L;
            if (segmentMarker == null) {
                LOGGER.warn("Unknown JPEG 2000 segment marker 0x{} at {}", (Object)Integer.toHexString(segmentMarkerCode), (Object)pos);
            } else {
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace(String.format("Found JPEG 2000 segment marker '%s' of length %d at %d", segmentMarker.getName(), segmentLength, pos));
                }
                switch (segmentMarker) {
                    case SOT: 
                    case SOD: 
                    case EOC: {
                        terminate = true;
                        break;
                    }
                    case SIZ: {
                        this.in.skipBytes(2);
                        this.codestreamSizeX = this.in.readInt();
                        LOGGER.trace("Read reference grid width {} at {}", this.codestreamSizeX, (Object)this.in.getFilePointer());
                        this.codestreamSizeY = this.in.readInt();
                        LOGGER.trace("Read reference grid height {} at {}", this.codestreamSizeY, (Object)this.in.getFilePointer());
                        this.in.skipBytes(24);
                        this.codestreamSizeC = this.in.readShort();
                        LOGGER.trace("Read total components {} at {}", this.codestreamSizeC, (Object)this.in.getFilePointer());
                        int type = this.in.readInt();
                        this.codestreamPixelType = this.convertPixelType(type);
                        LOGGER.trace("Read codestream pixel type {} at {}", this.codestreamPixelType, (Object)this.in.getFilePointer());
                        break;
                    }
                    case COD: {
                        this.in.skipBytes(5);
                        this.resolutionLevels = this.in.readUnsignedByte();
                        LOGGER.trace("Found number of resolution levels {} at {} ", this.resolutionLevels, (Object)this.in.getFilePointer());
                    }
                }
            }
            if (nextPos < 0L || nextPos >= maximumReadOffset || terminate) {
                LOGGER.trace("Exiting segment marker parse loop.");
                break;
            }
            LOGGER.trace("Seeking to next segment marker at {}", nextPos);
            this.in.seek(nextPos);
        }
    }

    public boolean isRawCodestream() {
        return this.isRawCodestream;
    }

    public Integer getResolutionLevels() {
        return this.resolutionLevels;
    }

    public Integer getHeaderSizeX() {
        return this.headerSizeX;
    }

    public Integer getHeaderSizeY() {
        return this.headerSizeY;
    }

    public Short getHeaderSizeC() {
        return this.headerSizeC;
    }

    public Integer getHeaderPixelType() {
        return this.headerPixelType;
    }

    public Integer getCodestreamSizeX() {
        return this.codestreamSizeX;
    }

    public Integer getCodestreamSizeY() {
        return this.codestreamSizeY;
    }

    public Short getCodestreamSizeC() {
        return this.codestreamSizeC;
    }

    public Integer getCodestreamPixelType() {
        return this.codestreamPixelType;
    }

    public int[][] getLookupTable() {
        return this.lut;
    }

    private int convertPixelType(int type) {
        if (type == 252117248 || type == 0xF070000) {
            return 3;
        }
        return 1;
    }
}

