/*
 * Decompiled with CFR 0.152.
 */
package loci.formats.tiff;

import java.io.IOException;
import java.util.Arrays;
import java.util.Vector;
import loci.common.DataTools;
import loci.common.LogTools;
import loci.common.RandomAccessInputStream;
import loci.common.Region;
import loci.formats.FormatException;
import loci.formats.codec.BitBuffer;
import loci.formats.codec.CodecOptions;
import loci.formats.tiff.IFD;
import loci.formats.tiff.IFDList;
import loci.formats.tiff.TiffCompression;
import loci.formats.tiff.TiffIFDEntry;
import loci.formats.tiff.TiffRational;

public class TiffParser {
    protected RandomAccessInputStream in;
    private byte[] cachedTileBuffer;

    public TiffParser(RandomAccessInputStream in) {
        this.in = in;
    }

    public RandomAccessInputStream getStream() {
        return this.in;
    }

    public boolean isValidHeader() {
        try {
            return this.checkHeader() != null;
        }
        catch (IOException e) {
            return false;
        }
    }

    public Boolean checkHeader() throws IOException {
        boolean bigEndian;
        if (this.in.length() < 4L) {
            return null;
        }
        this.in.seek(0L);
        int endianOne = this.in.read();
        int endianTwo = this.in.read();
        boolean littleEndian = endianOne == 73 && endianTwo == 73;
        boolean bl = bigEndian = endianOne == 77 && endianTwo == 77;
        if (!littleEndian && !bigEndian) {
            return null;
        }
        this.in.order(littleEndian);
        short magic = this.in.readShort();
        if (magic != 42 && magic != 43) {
            return null;
        }
        return new Boolean(littleEndian);
    }

    public IFDList getIFDs() throws IOException {
        return this.getIFDs(false);
    }

    public IFDList getIFDs(boolean skipThumbnails) throws IOException {
        return this.getIFDs(skipThumbnails, true);
    }

    public IFDList getIFDs(boolean skipThumbnails, boolean fillInEntries) throws IOException {
        IFD ifd;
        Boolean result = this.checkHeader();
        if (result == null) {
            return null;
        }
        this.in.seek(2L);
        boolean bigTiff = this.in.readShort() == 43;
        long offset = this.getFirstOffset(bigTiff);
        long ifdMax = (this.in.length() - 8L) / (long)(bigTiff ? 22 : 18);
        IFDList ifds = new IFDList();
        for (long ifdNum = 0L; ifdNum < ifdMax && (ifd = this.getIFD(ifdNum, offset, bigTiff, fillInEntries)) != null && ifd.size() > 2; ++ifdNum) {
            int subfileType;
            Number subfile = (Number)ifd.getIFDValue(254);
            int n = subfileType = subfile == null ? 0 : subfile.intValue();
            if (!skipThumbnails || subfileType == 0) {
                ifds.add(ifd);
            } else {
                ifd = null;
            }
            offset = this.getNextOffset(bigTiff, offset);
            if (offset > 0L && offset < this.in.length()) continue;
            if (offset == 0L) break;
            LogTools.debug("getIFDs: invalid IFD offset: " + offset);
            break;
        }
        return ifds;
    }

    public long[] getIFDOffsets() throws IOException {
        Boolean result = this.checkHeader();
        if (result == null) {
            return null;
        }
        this.in.seek(2L);
        boolean bigTiff = this.in.readShort() == 43;
        int bytesPerEntry = bigTiff ? 20 : 12;
        Vector<Long> offsets = new Vector<Long>();
        long offset = this.getFirstOffset(bigTiff);
        do {
            this.in.seek(offset);
            offsets.add(offset);
            short nEntries = this.in.readShort();
            this.in.skipBytes(nEntries * bytesPerEntry);
        } while ((offset = this.getNextOffset(bigTiff, offset)) > 0L && offset < this.in.length());
        long[] f = new long[offsets.size()];
        for (int i = 0; i < f.length; ++i) {
            f[i] = (Long)offsets.get(i);
        }
        return f;
    }

    public IFD getFirstIFD() throws IOException {
        Boolean result = this.checkHeader();
        if (result == null) {
            return null;
        }
        this.in.seek(2L);
        boolean bigTiff = this.in.readShort() == 43;
        long offset = this.getFirstOffset(bigTiff);
        IFD ifd = this.getIFD(0L, offset, bigTiff);
        if (ifd != null) {
            ifd.put(new Integer(1), new Boolean(bigTiff));
        }
        return ifd;
    }

    public TiffIFDEntry getFirstIFDEntry(int tag) throws IOException {
        Boolean result = this.checkHeader();
        if (result == null) {
            return null;
        }
        this.in.seek(2L);
        boolean bigTiff = this.in.readShort() == 43;
        long offset = this.getFirstOffset(bigTiff);
        this.in.seek(offset);
        long numEntries = bigTiff ? this.in.readLong() : (long)(this.in.readShort() & 0xFFFF);
        int i = 0;
        while ((long)i < numEntries) {
            this.in.seek(offset + 2L + (long)((bigTiff ? 20 : 12) * i));
            int entryTag = this.in.readShort() & 0xFFFF;
            if (entryTag == tag) {
                int valueCount;
                int entryType = this.in.readShort() & 0xFFFF;
                int n = valueCount = bigTiff ? (int)(this.in.readLong() & 0xFFFFFFFFFFFFFFFFL) : this.in.readInt();
                if (valueCount < 0) {
                    throw new RuntimeException("Count of '" + valueCount + "' unexpected.");
                }
                long valueOffset = this.getNextOffset(bigTiff, 0L);
                return new TiffIFDEntry(entryTag, entryType, valueCount, valueOffset);
            }
            ++i;
        }
        throw new IllegalArgumentException("Unknown tag: " + tag);
    }

    public long getFirstOffset() throws IOException {
        return this.getFirstOffset(false);
    }

    public long getFirstOffset(boolean bigTiff) throws IOException {
        if (bigTiff) {
            this.in.skipBytes(4);
        }
        return bigTiff ? this.in.readLong() : (long)this.in.readInt();
    }

    public IFD getIFD(long ifdNum, long offset) throws IOException {
        return this.getIFD(ifdNum, offset, false);
    }

    public IFD getIFD(long ifdNum, long offset, boolean bigTiff) throws IOException {
        return this.getIFD(ifdNum, offset, bigTiff, true);
    }

    public IFD getIFD(long ifdNum, long offset, boolean bigTiff, boolean fillInEntries) throws IOException {
        IFD ifd = new IFD();
        ifd.put(new Integer(0), new Boolean(this.in.isLittleEndian()));
        ifd.put(new Integer(1), new Boolean(bigTiff));
        LogTools.debug("getIFDs: seeking IFD #" + ifdNum + " at " + offset);
        this.in.seek(offset);
        long numEntries = bigTiff ? this.in.readLong() : (long)(this.in.readShort() & 0xFFFF);
        LogTools.debug("getIFDs: " + numEntries + " directory entries to read");
        if (numEntries == 0L || numEntries == 1L) {
            return ifd;
        }
        int bytesPerEntry = bigTiff ? 20 : 12;
        int baseOffset = bigTiff ? 8 : 2;
        int threshhold = bigTiff ? 8 : 4;
        int i = 0;
        while ((long)i < numEntries) {
            this.in.seek(offset + (long)baseOffset + (long)(bytesPerEntry * i));
            int tag = this.in.readShort() & 0xFFFF;
            int type = this.in.readShort() & 0xFFFF;
            int count = bigTiff ? (int)(this.in.readLong() & 0xFFFFFFFFFFFFFFFFL) : this.in.readInt();
            int bpe = IFD.getIFDTypeLength(type);
            LogTools.debug("getIFDs: read " + IFD.getIFDTagName(tag) + " (type=" + IFD.getIFDTypeName(type) + "; count=" + count + ")");
            if (count < 0 || bpe <= 0) {
                this.in.skipBytes(bytesPerEntry - 4 - (bigTiff ? 8 : 4));
            } else {
                long inputLen;
                Object value = null;
                long pointer = this.in.getFilePointer();
                if (count > threshhold / bpe) {
                    pointer = this.getNextOffset(bigTiff, 0L);
                }
                if ((long)(count * bpe) + pointer > (inputLen = this.in.length())) {
                    int oldCount = count;
                    count = (int)((inputLen - pointer) / (long)bpe);
                    LogTools.debug("getIFDs: truncated " + (oldCount - count) + " array elements for tag " + tag);
                }
                if (count < 0 || (long)count > this.in.length()) break;
                TiffIFDEntry entry = new TiffIFDEntry(tag, type, count, pointer);
                value = pointer != this.in.getFilePointer() && !fillInEntries ? entry : this.getIFDValue(entry);
                if (value != null && !ifd.containsKey(new Integer(tag))) {
                    ifd.put(new Integer(tag), value);
                }
            }
            ++i;
        }
        this.in.seek(offset + (long)baseOffset + (long)bytesPerEntry * numEntries);
        if (!(ifd.get(256) instanceof Number) || !(ifd.get(257) instanceof Number)) {
            return null;
        }
        return ifd;
    }

    public void fillInIFD(IFD ifd) throws IOException {
        Vector<TiffIFDEntry> entries = new Vector<TiffIFDEntry>();
        for (Integer key : ifd.keySet()) {
            if (!(ifd.get(key) instanceof TiffIFDEntry)) continue;
            entries.add((TiffIFDEntry)ifd.get(key));
        }
        Object[] e = entries.toArray(new TiffIFDEntry[entries.size()]);
        Arrays.sort(e);
        for (Object entry : e) {
            ifd.put(new Integer(((TiffIFDEntry)entry).getTag()), this.getIFDValue((TiffIFDEntry)entry));
        }
    }

    public Object getIFDValue(TiffIFDEntry entry) throws IOException {
        int type = entry.getType();
        int count = entry.getValueCount();
        long offset = entry.getValueOffset();
        if (offset != this.in.getFilePointer()) {
            this.in.seek(offset);
        }
        if (type == 1) {
            if (count == 1) {
                return new Short(this.in.readByte());
            }
            byte[] bytes = new byte[count];
            this.in.readFully(bytes);
            short[] shorts = new short[count];
            for (int j = 0; j < count; ++j) {
                shorts[j] = (short)(bytes[j] & 0xFF);
            }
            return shorts;
        }
        if (type == 2) {
            byte[] ascii = new byte[count];
            this.in.read(ascii);
            int nullCount = 0;
            for (int j = 0; j < count; ++j) {
                if (ascii[j] != 0 && j != count - 1) continue;
                ++nullCount;
            }
            String[] strings = nullCount == 1 ? null : new String[nullCount];
            Object s = null;
            int c = 0;
            int ndx = -1;
            for (int j = 0; j < count; ++j) {
                if (ascii[j] == 0) {
                    s = new String(ascii, ndx + 1, j - ndx - 1);
                    ndx = j;
                } else {
                    s = j == count - 1 ? new String(ascii, ndx + 1, j - ndx) : null;
                }
                if (strings == null || s == null) continue;
                strings[c++] = s;
            }
            return strings == null ? s : strings;
        }
        if (type == 3) {
            if (count == 1) {
                return new Integer(this.in.readShort() & 0xFFFF);
            }
            int[] shorts = new int[count];
            for (int j = 0; j < count; ++j) {
                shorts[j] = this.in.readShort() & 0xFFFF;
            }
            return shorts;
        }
        if (type == 4 || type == 13) {
            if (count == 1) {
                return new Long(this.in.readInt());
            }
            long[] longs = new long[count];
            for (int j = 0; j < count; ++j) {
                longs[j] = this.in.readInt();
            }
            return longs;
        }
        if (type == 16 || type == 17 || type == 18) {
            if (count == 1) {
                return new Long(this.in.readLong());
            }
            long[] longs = new long[count];
            for (int j = 0; j < count; ++j) {
                longs[j] = this.in.readLong();
            }
            return longs;
        }
        if (type == 5 || type == 10) {
            if (count == 1) {
                return new TiffRational(this.in.readInt(), this.in.readInt());
            }
            TiffRational[] rationals = new TiffRational[count];
            for (int j = 0; j < count; ++j) {
                rationals[j] = new TiffRational(this.in.readInt(), this.in.readInt());
            }
            return rationals;
        }
        if (type == 6 || type == 7) {
            if (count == 1) {
                return new Byte(this.in.readByte());
            }
            byte[] sbytes = new byte[count];
            this.in.read(sbytes);
            return sbytes;
        }
        if (type == 8) {
            if (count == 1) {
                return new Short(this.in.readShort());
            }
            short[] sshorts = new short[count];
            for (int j = 0; j < count; ++j) {
                sshorts[j] = this.in.readShort();
            }
            return sshorts;
        }
        if (type == 9) {
            if (count == 1) {
                return new Integer(this.in.readInt());
            }
            int[] slongs = new int[count];
            for (int j = 0; j < count; ++j) {
                slongs[j] = this.in.readInt();
            }
            return slongs;
        }
        if (type == 11) {
            if (count == 1) {
                return new Float(this.in.readFloat());
            }
            float[] floats = new float[count];
            for (int j = 0; j < count; ++j) {
                floats[j] = this.in.readFloat();
            }
            return floats;
        }
        if (type == 12) {
            if (count == 1) {
                return new Double(this.in.readDouble());
            }
            double[] doubles = new double[count];
            for (int j = 0; j < count; ++j) {
                doubles[j] = this.in.readDouble();
            }
            return doubles;
        }
        return null;
    }

    public String getComment() throws IOException {
        IFD firstIFD = this.getFirstIFD();
        return firstIFD == null ? null : firstIFD.getComment();
    }

    public byte[] getTile(IFD ifd, int row, int col) throws FormatException, IOException {
        int samplesPerPixel = ifd.getSamplesPerPixel();
        if (ifd.getPlanarConfiguration() == 2) {
            samplesPerPixel = 1;
        }
        int bpp = ifd.getBytesPerSample()[0];
        int width = (int)ifd.getTileWidth();
        int height = (int)ifd.getTileLength();
        byte[] buf = new byte[width * height * samplesPerPixel * bpp];
        return this.getTile(ifd, buf, row, col);
    }

    public byte[] getTile(IFD ifd, byte[] buf, int row, int col) throws FormatException, IOException {
        int size;
        byte[] jpegTable = (byte[])ifd.getIFDValue(347, false, null);
        CodecOptions options = new CodecOptions();
        options.interleaved = true;
        options.littleEndian = ifd.isLittleEndian();
        long tileWidth = ifd.getTileWidth();
        long tileLength = ifd.getTileLength();
        int samplesPerPixel = ifd.getSamplesPerPixel();
        int planarConfig = ifd.getPlanarConfiguration();
        int compression = ifd.getCompression();
        long numTileCols = ifd.getTilesPerRow();
        int pixel = ifd.getBytesPerSample()[0];
        int effectiveChannels = planarConfig == 2 ? 1 : samplesPerPixel;
        long[] stripOffsets = ifd.getStripOffsets();
        long[] stripByteCounts = ifd.getStripByteCounts();
        int tileNumber = (int)((long)row * numTileCols + (long)col);
        byte[] tile = new byte[(int)stripByteCounts[tileNumber]];
        this.in.seek(stripOffsets[tileNumber]);
        this.in.read(tile);
        options.maxBytes = size = (int)(tileWidth * tileLength * (long)pixel * (long)effectiveChannels);
        if (jpegTable != null) {
            byte[] q = new byte[jpegTable.length + tile.length - 4];
            System.arraycopy(jpegTable, 0, q, 0, jpegTable.length - 2);
            System.arraycopy(tile, 2, q, jpegTable.length - 2, tile.length - 2);
            tile = TiffCompression.uncompress(q, compression, options);
        } else {
            tile = TiffCompression.uncompress(tile, compression, options);
        }
        TiffCompression.undifference(tile, ifd);
        TiffParser.unpackBytes(buf, 0, tile, ifd);
        return buf;
    }

    public byte[][] getSamples(IFD ifd) throws FormatException, IOException {
        return this.getSamples(ifd, 0, 0, (int)ifd.getImageWidth(), (int)ifd.getImageLength());
    }

    public byte[][] getSamples(IFD ifd, int x, int y, int w, int h) throws FormatException, IOException {
        int samplesPerPixel = ifd.getSamplesPerPixel();
        int bpp = ifd.getBytesPerSample()[0];
        long width = ifd.getImageWidth();
        long length = ifd.getImageLength();
        byte[] b = new byte[w * h * samplesPerPixel * bpp];
        this.getSamples(ifd, b, x, y, w, h);
        byte[][] samples = new byte[samplesPerPixel][w * h * bpp];
        for (int i = 0; i < samplesPerPixel; ++i) {
            System.arraycopy(b, i * w * h * bpp, samples[i], 0, samples[i].length);
        }
        b = null;
        return samples;
    }

    public byte[] getSamples(IFD ifd, byte[] buf) throws FormatException, IOException {
        long width = ifd.getImageWidth();
        long length = ifd.getImageLength();
        return this.getSamples(ifd, buf, 0, 0, width, length);
    }

    public byte[] getSamples(IFD ifd, byte[] buf, int x, int y, long width, long height) throws FormatException, IOException {
        LogTools.debug("parsing IFD entries");
        boolean littleEndian = ifd.isLittleEndian();
        this.in.order(littleEndian);
        int samplesPerPixel = ifd.getSamplesPerPixel();
        long tileWidth = ifd.getTileWidth();
        long tileLength = ifd.getTileLength();
        if (tileLength <= 0L) {
            LogTools.debug("Tile length is " + tileLength + "; setting it to " + height);
            tileLength = height;
        }
        long numTileRows = ifd.getTilesPerColumn();
        long numTileCols = ifd.getTilesPerRow();
        int photoInterp = ifd.getPhotometricInterpretation();
        int planarConfig = ifd.getPlanarConfiguration();
        int pixel = ifd.getBytesPerSample()[0];
        int effectiveChannels = planarConfig == 2 ? 1 : samplesPerPixel;
        ifd.printIFD();
        if (width * height > Integer.MAX_VALUE) {
            throw new FormatException("Sorry, ImageWidth x ImageLength > 2147483647 is not supported (" + width + " x " + height + ")");
        }
        if (width * height * (long)effectiveChannels * (long)pixel > Integer.MAX_VALUE) {
            throw new FormatException("Sorry, ImageWidth x ImageLength x SamplesPerPixel x BitsPerSample > 2147483647 is not supported (" + width + " x " + height + " x " + samplesPerPixel + " x " + pixel * 8 + ")");
        }
        int numSamples = (int)(width * height);
        LogTools.debug("reading image data (samplesPerPixel=" + samplesPerPixel + "; numSamples=" + numSamples + ")");
        int compression = ifd.getCompression();
        if ((long)x % tileWidth == 0L && (long)y % tileLength == 0L && width == tileWidth && height == tileLength && samplesPerPixel == 1 && ifd.getBitsPerSample()[0] % 8 == 0 && photoInterp != 0 && photoInterp != 5 && photoInterp != 6 && compression == 1) {
            long[] stripOffsets = ifd.getStripOffsets();
            long[] stripByteCounts = ifd.getStripByteCounts();
            int tile = (int)((long)y / tileLength * numTileCols + (long)x / tileWidth);
            this.in.seek(stripOffsets[tile]);
            this.in.read(buf, 0, (int)Math.min((long)buf.length, stripByteCounts[tile]));
            return buf;
        }
        long nrows = numTileRows;
        if (planarConfig == 2) {
            numTileRows *= (long)samplesPerPixel;
        }
        Region imageBounds = new Region(x, y, (int)width, (int)(height * (long)(samplesPerPixel / effectiveChannels)));
        int endX = (int)width + x;
        int endY = (int)height + y;
        int rowLen = pixel * (int)tileWidth;
        int tileSize = (int)((long)rowLen * tileLength);
        int planeSize = (int)(width * height * (long)pixel);
        int outputRowLen = (int)((long)pixel * width);
        int bufferSizeSamplesPerPixel = samplesPerPixel;
        if (ifd.getPlanarConfiguration() == 2) {
            bufferSizeSamplesPerPixel = 1;
        }
        int bpp = ifd.getBytesPerSample()[0];
        int bufferSize = (int)tileWidth * (int)tileLength * bufferSizeSamplesPerPixel * bpp;
        if (this.cachedTileBuffer == null || this.cachedTileBuffer.length != bufferSize) {
            this.cachedTileBuffer = new byte[bufferSize];
        }
        Region tileBounds = new Region(0, 0, (int)tileWidth, (int)tileLength);
        int row = 0;
        while ((long)row < numTileRows) {
            int col = 0;
            while ((long)col < numTileCols) {
                tileBounds.x = col * (int)tileWidth;
                tileBounds.y = row * (int)tileLength;
                if (imageBounds.intersects(tileBounds)) {
                    if (planarConfig == 2) {
                        tileBounds.y = (int)((long)row % nrows * tileLength);
                    }
                    this.getTile(ifd, this.cachedTileBuffer, row, col);
                    int tileX = Math.max(tileBounds.x, x);
                    int tileY = Math.max(tileBounds.y, y);
                    int realX = tileX % (int)tileWidth;
                    int realY = tileY % (int)tileLength;
                    int twidth = (int)Math.min((long)(endX - tileX), tileWidth - (long)realX);
                    int theight = (int)Math.min((long)(endY - tileY), tileLength - (long)realY);
                    int copy = pixel * twidth;
                    realX *= pixel;
                    realY *= rowLen;
                    for (int q = 0; q < effectiveChannels; ++q) {
                        int src = q * tileSize + realX + realY;
                        int dest = q * planeSize + pixel * (tileX - x) + outputRowLen * (tileY - y);
                        if (planarConfig == 2) {
                            dest = (int)((long)dest + (long)planeSize * ((long)row / nrows));
                        }
                        for (int tileRow = 0; tileRow < theight; ++tileRow) {
                            System.arraycopy(this.cachedTileBuffer, src, buf, dest, copy);
                            src += rowLen;
                            dest += outputRowLen;
                        }
                    }
                }
                ++col;
            }
            ++row;
        }
        return buf;
    }

    public static void planarUnpack(byte[] samples, int startIndex, byte[] bytes, IFD ifd) throws FormatException {
        int numBytes;
        BitBuffer bb = new BitBuffer(bytes);
        int realBytes = numBytes = ifd.getBytesPerSample()[0];
        if (numBytes == 3) {
            ++numBytes;
        }
        int bitsPerSample = ifd.getBitsPerSample()[0];
        boolean littleEndian = ifd.isLittleEndian();
        int photoInterp = ifd.getPhotometricInterpretation();
        for (int j = 0; j < bytes.length / realBytes; ++j) {
            int value = bb.getBits(bitsPerSample);
            if (littleEndian) {
                value = DataTools.swap(value) >> 32 - bitsPerSample;
            }
            if (photoInterp == 0) {
                value = (int)(Math.pow(2.0, bitsPerSample) - 1.0 - (double)value);
            } else if (photoInterp == 5) {
                value = Integer.MAX_VALUE - value;
            }
            if (numBytes * (startIndex + j) >= samples.length) continue;
            DataTools.unpackBytes(value, samples, numBytes * (startIndex + j), numBytes, littleEndian);
        }
    }

    public static void unpackBytes(byte[] samples, int startIndex, byte[] bytes, IFD ifd) throws FormatException {
        if (ifd.getPlanarConfiguration() == 2) {
            TiffParser.planarUnpack(samples, startIndex, bytes, ifd);
            return;
        }
        int compression = ifd.getCompression();
        int photoInterp = ifd.getPhotometricInterpretation();
        if (compression == 7) {
            photoInterp = 2;
        }
        int[] bitsPerSample = ifd.getBitsPerSample();
        int nChannels = bitsPerSample.length;
        int nSamples = samples.length / nChannels;
        int totalBits = 0;
        for (int i = 0; i < nChannels; ++i) {
            totalBits += bitsPerSample[i];
        }
        int sampleCount = 8 * bytes.length / totalBits;
        if (photoInterp == 6) {
            sampleCount *= 3;
        }
        LogTools.debug("unpacking " + sampleCount + " samples (startIndex=" + startIndex + "; totalBits=" + totalBits + "; numBytes=" + bytes.length + ")");
        long imageWidth = ifd.getImageWidth();
        int bps0 = bitsPerSample[0];
        int numBytes = ifd.getBytesPerSample()[0];
        boolean noDiv8 = bps0 % 8 != 0;
        boolean bps8 = bps0 == 8;
        boolean bps16 = bps0 == 16;
        int row = startIndex / (int)imageWidth;
        boolean col = false;
        boolean cw = false;
        boolean ch = false;
        boolean littleEndian = ifd.isLittleEndian();
        int[] reference = ifd.getIFDIntArray(532, false);
        int[] subsampling = ifd.getIFDIntArray(530, false);
        TiffRational[] coefficients = (TiffRational[])ifd.getIFDValue(529);
        boolean count = false;
        BitBuffer bb = new BitBuffer(bytes);
        if ((bps8 || bps16) && bytes.length <= samples.length && nChannels == 1 && photoInterp != 0 && photoInterp != 5 && photoInterp != 6) {
            System.arraycopy(bytes, 0, samples, 0, bytes.length);
            return;
        }
        block1: for (int j = 0; j < sampleCount; ++j) {
            for (int i = 0; i < nChannels; ++i) {
                int index = numBytes * (j * nChannels + i);
                int ndx = startIndex + j;
                if (ndx >= nSamples) continue block1;
                int outputIndex = i * nSamples + ndx * numBytes;
                if (noDiv8) {
                    short s = 0;
                    if (i == 0 && photoInterp == 3 || photoInterp != 32803 && photoInterp != 3) {
                        s = (short)(bb.getBits(bps0) & 0xFFFF);
                        if ((long)ndx % imageWidth == imageWidth - 1L && bps0 < 8) {
                            bb.skipBits(imageWidth * (long)bps0 * (long)sampleCount % 8L);
                        }
                    }
                    if (photoInterp == 0 || photoInterp == 5) {
                        s = (short)(Math.pow(2.0, bitsPerSample[0]) - 1.0 - (double)s);
                    }
                    if (outputIndex + numBytes > samples.length) continue;
                    DataTools.unpackBytes(s, samples, outputIndex, numBytes, littleEndian);
                    continue;
                }
                if (bps8) {
                    if (outputIndex >= samples.length) continue block1;
                    if (photoInterp != 6) {
                        samples[outputIndex] = (byte)(bytes[index] & 0xFF);
                    }
                    if (photoInterp == 0) {
                        samples[outputIndex] = (byte)(255 - samples[outputIndex]);
                        continue;
                    }
                    if (photoInterp == 5) {
                        samples[outputIndex] = (byte)(Integer.MAX_VALUE - samples[outputIndex]);
                        continue;
                    }
                    if (photoInterp != 6 || i != bitsPerSample.length - 1) continue;
                    float lumaRed = 0.299f;
                    float lumaGreen = 0.587f;
                    float lumaBlue = 0.114f;
                    if (coefficients != null) {
                        lumaRed = coefficients[0].floatValue();
                        lumaGreen = coefficients[1].floatValue();
                        lumaBlue = coefficients[2].floatValue();
                    }
                    int subX = subsampling == null ? 2 : subsampling[0];
                    int subY = subsampling == null ? 2 : subsampling[1];
                    int block = subX * subY;
                    int lumaIndex = j + 2 * (j / block);
                    int chromaIndex = j / block * (block + 2) + block;
                    if (chromaIndex + 1 >= bytes.length) continue block1;
                    int tile = ndx / block;
                    int nTiles = (int)(imageWidth / (long)subX);
                    int pixel = ndx % block;
                    long r = subY * (tile / nTiles) + pixel / subX;
                    long c = subX * (tile % nTiles) + pixel % subX;
                    int idx = (int)(r * imageWidth + c);
                    if (idx >= nSamples) continue;
                    int y = (bytes[lumaIndex] & 0xFF) - reference[0];
                    int cb = (bytes[chromaIndex] & 0xFF) - reference[2];
                    int cr = (bytes[chromaIndex + 1] & 0xFF) - reference[4];
                    int red = (int)((float)cr * (2.0f - 2.0f * lumaRed) + (float)y);
                    int blue = (int)((float)cb * (2.0f - 2.0f * lumaBlue) + (float)y);
                    int green = (int)(((float)y - lumaBlue * (float)blue - lumaRed * (float)red) / lumaGreen);
                    samples[idx] = (byte)red;
                    samples[nSamples + idx] = (byte)green;
                    samples[2 * nSamples + idx] = (byte)blue;
                    continue;
                }
                int offset = numBytes + index < bytes.length ? index : bytes.length - numBytes;
                long v = DataTools.bytesToLong(bytes, offset, numBytes, littleEndian);
                if (photoInterp == 0) {
                    long max = (long)Math.pow(2.0, numBytes * 8) - 1L;
                    v = max - v;
                } else if (photoInterp == 5) {
                    v = Integer.MAX_VALUE - v;
                }
                if (ndx * numBytes >= nSamples) continue block1;
                DataTools.unpackBytes(v, samples, i * nSamples + ndx * numBytes, numBytes, littleEndian);
            }
        }
    }

    long getNextOffset(boolean bigTiff, long previous) throws IOException {
        if (bigTiff) {
            return this.in.readLong();
        }
        long offset = previous & 0xFFFFFFFF00000000L | (long)this.in.readInt() & 0xFFFFFFFFL;
        if (offset < previous) {
            offset += 0x100000000L;
        }
        return offset;
    }
}

