/*
 * Decompiled with CFR 0.152.
 */
package loci.formats.in;

import java.io.File;
import java.io.IOException;
import java.util.Hashtable;
import java.util.Stack;
import java.util.Vector;
import loci.common.DataTools;
import loci.common.Location;
import loci.common.RandomAccessInputStream;
import loci.formats.CoreMetadata;
import loci.formats.FormatException;
import loci.formats.FormatReader;
import loci.formats.FormatTools;
import loci.formats.MetadataTools;
import loci.formats.TiffTools;
import loci.formats.in.MDBParser;
import loci.formats.meta.FilterMetadata;
import loci.formats.meta.MetadataStore;

public class ZeissLSMReader
extends FormatReader {
    public static final String[] MDB_SUFFIX = new String[]{"mdb"};
    private static final int ZEISS_ID = 34412;
    private static final int TYPE_SUBBLOCK = 0;
    private static final int TYPE_ASCII = 2;
    private static final int TYPE_LONG = 4;
    private static final int TYPE_RATIONAL = 5;
    private static final int SUBBLOCK_RECORDING = 0x10000000;
    private static final int SUBBLOCK_LASERS = 0x30000000;
    private static final int SUBBLOCK_LASER = 0x50000000;
    private static final int SUBBLOCK_TRACKS = 0x20000000;
    private static final int SUBBLOCK_TRACK = 0x40000000;
    private static final int SUBBLOCK_DETECTION_CHANNELS = 0x60000000;
    private static final int SUBBLOCK_DETECTION_CHANNEL = 0x70000000;
    private static final int SUBBLOCK_ILLUMINATION_CHANNELS = Integer.MIN_VALUE;
    private static final int SUBBLOCK_ILLUMINATION_CHANNEL = -1879048192;
    private static final int SUBBLOCK_BEAM_SPLITTERS = -1610612736;
    private static final int SUBBLOCK_BEAM_SPLITTER = -1342177280;
    private static final int SUBBLOCK_DATA_CHANNELS = -1073741824;
    private static final int SUBBLOCK_DATA_CHANNEL = -805306368;
    private static final int SUBBLOCK_TIMERS = 0x11000000;
    private static final int SUBBLOCK_TIMER = 0x12000000;
    private static final int SUBBLOCK_MARKERS = 0x13000000;
    private static final int SUBBLOCK_MARKER = 0x14000000;
    private static final int SUBBLOCK_END = -1;
    private static final int SUBBLOCK_GAMMA = 1;
    private static final int SUBBLOCK_BRIGHTNESS = 2;
    private static final int SUBBLOCK_CONTRAST = 3;
    private static final int SUBBLOCK_RAMP = 4;
    private static final int SUBBLOCK_KNOTS = 5;
    private static final int SUBBLOCK_PALETTE = 6;
    private static final int RECORDING_ENTRY_DESCRIPTION = 0x10000002;
    private static final int RECORDING_ENTRY_OBJECTIVE = 0x10000004;
    private static final int TRACK_ENTRY_TIME_BETWEEN_STACKS = 0x4000000B;
    private static final int LASER_ENTRY_NAME = 0x50000001;
    private static final int CHANNEL_ENTRY_DETECTOR_GAIN = 0x70000003;
    private static final int CHANNEL_ENTRY_PINHOLE_DIAMETER = 0x70000009;
    private static final int CHANNEL_ENTRY_SPI_WAVELENGTH_START = 0x70000022;
    private static final int CHANNEL_ENTRY_SPI_WAVELENGTH_END = 1879048227;
    private static final int ILLUM_CHANNEL_WAVELENGTH = -1879048189;
    private static final int START_TIME = 268435510;
    private static final int DATA_CHANNEL_NAME = -805306367;
    private static final int TEXT = 13;
    private static final int LINE = 14;
    private static final int SCALE_BAR = 15;
    private static final int OPEN_ARROW = 16;
    private static final int CLOSED_ARROW = 17;
    private static final int RECTANGLE = 18;
    private static final int ELLIPSE = 19;
    private static final int CLOSED_POLYLINE = 20;
    private static final int OPEN_POLYLINE = 21;
    private static final int CLOSED_BEZIER = 22;
    private static final int OPEN_BEZIER = 23;
    private static final int CIRCLE = 24;
    private static final int PALETTE = 25;
    private static final int POLYLINE_ARROW = 26;
    private static final int BEZIER_WITH_ARROW = 27;
    private static final int ANGLE = 28;
    private static final int CIRCLE_3POINT = 29;
    private static Hashtable metadataKeys = ZeissLSMReader.createKeys();
    private double pixelSizeX;
    private double pixelSizeY;
    private double pixelSizeZ;
    private byte[][] lut = null;
    private Vector timestamps;
    private int validChannels;
    private String[] lsmFilenames;
    private Hashtable[][] ifds;

    public ZeissLSMReader() {
        super("Zeiss Laser-Scanning Microscopy", new String[]{"lsm", "mdb"});
        this.blockCheckLen = 4;
    }

    public void close() throws IOException {
        super.close();
        this.pixelSizeZ = 0.0;
        this.pixelSizeY = 0.0;
        this.pixelSizeX = 0.0;
        this.lut = null;
        this.timestamps = null;
        this.validChannels = 0;
        this.lsmFilenames = null;
        this.ifds = null;
    }

    public boolean isThisType(RandomAccessInputStream stream) throws IOException {
        if (!FormatTools.validStream(stream, this.blockCheckLen, false)) {
            return false;
        }
        byte[] check = new byte[this.blockCheckLen];
        stream.read(check);
        return TiffTools.isValidHeader(check) || check[2] == 83 && check[3] == 116;
    }

    public String[] getUsedFiles(boolean noPixels) {
        FormatTools.assertId(this.currentId, true, 1);
        if (noPixels) {
            if (ZeissLSMReader.checkSuffix(this.currentId, MDB_SUFFIX)) {
                return new String[]{this.currentId};
            }
            return null;
        }
        if (this.lsmFilenames == null) {
            return new String[]{this.currentId};
        }
        if (this.lsmFilenames.length == 1 && this.currentId.equals(this.lsmFilenames[0])) {
            return this.lsmFilenames;
        }
        String[] files = new String[this.lsmFilenames.length + 1];
        System.arraycopy(this.lsmFilenames, 0, files, 0, this.lsmFilenames.length);
        files[files.length - 1] = this.currentId;
        return files;
    }

    public byte[][] get8BitLookupTable() throws FormatException, IOException {
        FormatTools.assertId(this.currentId, true, 1);
        if (this.lut == null || this.lut[this.getSeries()] == null || this.getPixelType() != 1) {
            return null;
        }
        byte[][] b = new byte[3][256];
        for (int i = 2; i >= 3 - this.validChannels; --i) {
            for (int j = 0; j < 256; ++j) {
                b[i][j] = (byte)j;
            }
        }
        return b;
    }

    public short[][] get16BitLookupTable() throws FormatException, IOException {
        FormatTools.assertId(this.currentId, true, 1);
        if (this.lut == null || this.lut[this.getSeries()] == null || this.getPixelType() != 3) {
            return null;
        }
        short[][] s = new short[3][65536];
        for (int i = 2; i >= 3 - this.validChannels; --i) {
            for (int j = 0; j < s[i].length; ++j) {
                s[i][j] = (short)j;
            }
        }
        return s;
    }

    public byte[] openBytes(int no, byte[] buf, int x, int y, int w, int h) throws FormatException, IOException {
        FormatTools.assertId(this.currentId, true, 1);
        FormatTools.checkPlaneNumber(this, no);
        FormatTools.checkBufferSize(this, buf.length, w, h);
        this.in = new RandomAccessInputStream(this.lsmFilenames[this.getSeries()]);
        this.in.order(!this.isLittleEndian());
        TiffTools.getSamples(this.ifds[this.getSeries()][no], this.in, buf, x, y, w, h);
        this.in.close();
        return buf;
    }

    protected void initFile(String id) throws FormatException, IOException {
        this.debug("ZeissLSMReader.initFile(" + id + ")");
        super.initFile(id);
        if (!ZeissLSMReader.checkSuffix(id, MDB_SUFFIX)) {
            Location parentFile = new Location(id).getAbsoluteFile().getParentFile();
            String[] fileList = parentFile.list();
            for (int i = 0; i < fileList.length; ++i) {
                Location file2;
                if (!ZeissLSMReader.checkSuffix(fileList[i], MDB_SUFFIX) || (file2 = new Location(parentFile, fileList[i]).getAbsoluteFile()).isDirectory()) continue;
                this.setId(new Location(parentFile, fileList[i]).getAbsolutePath());
                return;
            }
            this.lsmFilenames = new String[]{id};
        } else {
            this.lsmFilenames = this.parseMDB(id);
        }
        if (this.lsmFilenames.length == 0) {
            throw new FormatException("LSM files were not found.");
        }
        this.timestamps = new Vector();
        this.core = new CoreMetadata[this.lsmFilenames.length];
        this.ifds = new Hashtable[this.core.length][];
        for (int i = 0; i < this.core.length; ++i) {
            this.core[i] = new CoreMetadata();
            RandomAccessInputStream s = new RandomAccessInputStream(this.lsmFilenames[i]);
            this.core[i].littleEndian = s.read() == 73;
            s.order(this.isLittleEndian());
            s.seek(0L);
            this.ifds[i] = TiffTools.getIFDs(s);
            s.close();
        }
        this.status("Removing thumbnails");
        FilterMetadata store = new FilterMetadata(this.getMetadataStore(), this.isMetadataFiltered());
        for (int series = 0; series < this.ifds.length; ++series) {
            int i;
            Vector<Hashtable> newIFDs = new Vector<Hashtable>();
            for (i = 0; i < this.ifds[series].length; ++i) {
                Hashtable ifd = this.ifds[series][i];
                long subFileType = TiffTools.getIFDLongValue(ifd, 254, false, 0L);
                if (subFileType != 0L) continue;
                if (TiffTools.getCompression(ifd) != 5) {
                    ifd.put(new Integer(317), new Integer(1));
                }
                newIFDs.add(ifd);
            }
            this.ifds[series] = newIFDs.toArray(new Hashtable[0]);
            for (i = 1; i < this.ifds[series].length; ++i) {
                long thisOffset = TiffTools.getStripOffsets(this.ifds[series][i])[0] & 0xFFFFFFFFL;
                long prevOffset = TiffTools.getStripOffsets(this.ifds[series][i - 1])[0];
                if (prevOffset < 0L) {
                    prevOffset &= 0xFFFFFFFFL;
                }
                if (prevOffset <= thisOffset) continue;
                this.ifds[series][i].put(new Integer(273), new Long(thisOffset += 0xFFFFFFFFL));
            }
            this.initMetadata(series);
            this.core[series].littleEndian = !this.isLittleEndian();
            store.setPixelsBigEndian(new Boolean(!this.isLittleEndian()), series, 0);
        }
        this.setSeries(0);
    }

    protected void initMetadata(int series) throws FormatException, IOException {
        int i;
        this.setSeries(series);
        Hashtable ifd = this.ifds[series][0];
        this.in = new RandomAccessInputStream(this.lsmFilenames[series]);
        this.in.order(this.isLittleEndian());
        int photo = TiffTools.getPhotometricInterpretation(ifd);
        int samples = TiffTools.getSamplesPerPixel(ifd);
        this.core[series].sizeX = (int)TiffTools.getImageWidth(ifd);
        this.core[series].sizeY = (int)TiffTools.getImageLength(ifd);
        this.core[series].rgb = samples > 1 || photo == 2;
        this.core[series].interleaved = false;
        this.core[series].sizeC = this.isRGB() ? samples : 1;
        this.core[series].pixelType = TiffTools.getPixelType(ifd);
        this.core[series].imageCount = this.ifds[series].length;
        this.core[series].sizeZ = this.getImageCount();
        this.core[series].sizeT = 1;
        String keyPrefix = "Series " + series + " ";
        this.status("Reading LSM metadata for series #" + series);
        FilterMetadata store = new FilterMetadata(this.getMetadataStore(), this.isMetadataFiltered());
        store.setInstrumentID("Instrument:" + series, series);
        store.setImageInstrumentRef("Instrument:" + series, series);
        short[] s = TiffTools.getIFDShortArray(ifd, 34412, true);
        byte[] cz = new byte[s.length];
        for (int i2 = 0; i2 < s.length; ++i2) {
            cz[i2] = (byte)s[i2];
        }
        RandomAccessInputStream ras = new RandomAccessInputStream(cz);
        ras.order(this.isLittleEndian());
        this.addMeta(keyPrefix + "MagicNumber ", ras.readInt());
        this.addMeta(keyPrefix + "StructureSize", ras.readInt());
        this.addMeta(keyPrefix + "DimensionX", ras.readInt());
        this.addMeta(keyPrefix + "DimensionY", ras.readInt());
        this.core[series].sizeZ = ras.readInt();
        ras.skipBytes(4);
        this.core[series].sizeT = ras.readInt();
        int dataType = ras.readInt();
        switch (dataType) {
            case 2: {
                this.addMeta(keyPrefix + "DataType", "12 bit unsigned integer");
                break;
            }
            case 5: {
                this.addMeta(keyPrefix + "DataType", "32 bit float");
                break;
            }
            case 0: {
                this.addMeta(keyPrefix + "DataType", "varying data types");
                break;
            }
            default: {
                this.addMeta(keyPrefix + "DataType", "8 bit unsigned integer");
            }
        }
        this.addMeta(keyPrefix + "ThumbnailX", ras.readInt());
        this.addMeta(keyPrefix + "ThumbnailY", ras.readInt());
        this.pixelSizeX = ras.readDouble() * 1000000.0;
        this.pixelSizeY = ras.readDouble() * 1000000.0;
        this.pixelSizeZ = ras.readDouble() * 1000000.0;
        this.addMeta(keyPrefix + "VoxelSizeX", new Double(this.pixelSizeX));
        this.addMeta(keyPrefix + "VoxelSizeY", new Double(this.pixelSizeY));
        this.addMeta(keyPrefix + "VoxelSizeZ", new Double(this.pixelSizeZ));
        this.addMeta(keyPrefix + "OriginX", ras.readDouble());
        this.addMeta(keyPrefix + "OriginY", ras.readDouble());
        this.addMeta(keyPrefix + "OriginZ", ras.readDouble());
        short scanType = ras.readShort();
        switch (scanType) {
            case 0: {
                this.addMeta(keyPrefix + "ScanType", "x-y-z scan");
                this.core[series].dimensionOrder = "XYZCT";
                break;
            }
            case 1: {
                this.addMeta(keyPrefix + "ScanType", "z scan (x-z plane)");
                this.core[series].dimensionOrder = "XYZCT";
                break;
            }
            case 2: {
                this.addMeta(keyPrefix + "ScanType", "line scan");
                this.core[series].dimensionOrder = "XYZCT";
                break;
            }
            case 3: {
                this.addMeta(keyPrefix + "ScanType", "time series x-y");
                this.core[series].dimensionOrder = "XYTCZ";
                break;
            }
            case 4: {
                this.addMeta(keyPrefix + "ScanType", "time series x-z");
                this.core[series].dimensionOrder = "XYZTC";
                break;
            }
            case 5: {
                this.addMeta(keyPrefix + "ScanType", "time series 'Mean of ROIs'");
                this.core[series].dimensionOrder = "XYTCZ";
                break;
            }
            case 6: {
                this.addMeta(keyPrefix + "ScanType", "time series x-y-z");
                this.core[series].dimensionOrder = "XYZTC";
                break;
            }
            case 7: {
                this.addMeta(keyPrefix + "ScanType", "spline scan");
                this.core[series].dimensionOrder = "XYCTZ";
                break;
            }
            case 8: {
                this.addMeta(keyPrefix + "ScanType", "spline scan x-z");
                this.core[series].dimensionOrder = "XYCZT";
                break;
            }
            case 9: {
                this.addMeta(keyPrefix + "ScanType", "time series spline plane x-z");
                this.core[series].dimensionOrder = "XYTCZ";
                break;
            }
            case 10: {
                this.addMeta(keyPrefix + "ScanType", "point mode");
                this.core[series].dimensionOrder = "XYZCT";
                break;
            }
            default: {
                this.addMeta(keyPrefix + "ScanType", "x-y-z scan");
                this.core[series].dimensionOrder = "XYZCT";
            }
        }
        boolean bl = this.core[series].indexed = this.lut != null && this.lut[series] != null && this.getSizeC() == 1;
        if (this.isIndexed()) {
            this.core[series].sizeC = 1;
            this.core[series].rgb = false;
        }
        if (this.getSizeC() == 0) {
            this.core[series].sizeC = 1;
        }
        if (this.isRGB()) {
            this.core[series].dimensionOrder = this.getDimensionOrder().replaceAll("C", "");
            this.core[series].dimensionOrder = this.getDimensionOrder().replaceAll("XY", "XYC");
        }
        if (this.isIndexed()) {
            this.core[series].rgb = false;
        }
        this.core[series].imageCount = this.getEffectiveSizeC() == 0 ? this.getSizeZ() * this.getSizeT() : this.getSizeZ() * this.getSizeT() * this.getEffectiveSizeC();
        if (this.getImageCount() != this.ifds[series].length) {
            int diff = this.getImageCount() - this.ifds[series].length;
            this.core[series].imageCount = this.ifds[series].length;
            if (diff % this.getSizeZ() == 0) {
                this.core[series].sizeT -= diff / this.getSizeZ();
            } else if (diff % this.getSizeT() == 0) {
                this.core[series].sizeZ -= diff / this.getSizeT();
            } else if (this.getSizeZ() > 1) {
                this.core[series].sizeZ = this.ifds[series].length;
                this.core[series].sizeT = 1;
            } else if (this.getSizeT() > 1) {
                this.core[series].sizeT = this.ifds[series].length;
                this.core[series].sizeZ = 1;
            }
        }
        if (this.getSizeZ() == 0) {
            this.core[series].sizeZ = this.getImageCount();
        }
        if (this.getSizeT() == 0) {
            this.core[series].sizeT = this.getImageCount() / this.getSizeZ();
        }
        MetadataTools.populatePixels(store, this, true);
        MetadataTools.setDefaultCreationDate(store, this.getCurrentFile(), series);
        short spectralScan = ras.readShort();
        if (spectralScan != 1) {
            this.addMeta(keyPrefix + "SpectralScan", "no spectral scan");
        } else {
            this.addMeta(keyPrefix + "SpectralScan", "acquired with spectral scan");
        }
        int type = ras.readInt();
        switch (type) {
            case 1: {
                this.addMeta(keyPrefix + "DataType2", "calculated data");
                break;
            }
            case 2: {
                this.addMeta(keyPrefix + "DataType2", "animation");
                break;
            }
            default: {
                this.addMeta(keyPrefix + "DataType2", "original scan data");
            }
        }
        long[] overlayOffsets = new long[9];
        String[] overlayKeys = new String[]{"VectorOverlay", "InputLut", "OutputLut", "ROI", "BleachROI", "MeanOfRoisOverlay", "TopoIsolineOverlay", "TopoProfileOverlay", "LinescanOverlay"};
        overlayOffsets[0] = ras.readInt();
        overlayOffsets[1] = ras.readInt();
        overlayOffsets[2] = ras.readInt();
        long channelColorsOffset = ras.readInt();
        this.addMeta(keyPrefix + "TimeInterval", ras.readDouble());
        ras.skipBytes(4);
        long scanInformationOffset = ras.readInt();
        ras.skipBytes(4);
        long timeStampOffset = ras.readInt();
        long eventListOffset = ras.readInt();
        overlayOffsets[3] = ras.readInt();
        overlayOffsets[4] = ras.readInt();
        ras.skipBytes(4);
        this.addMeta(keyPrefix + "DisplayAspectX", ras.readDouble());
        this.addMeta(keyPrefix + "DisplayAspectY", ras.readDouble());
        this.addMeta(keyPrefix + "DisplayAspectZ", ras.readDouble());
        this.addMeta(keyPrefix + "DisplayAspectTime", ras.readDouble());
        overlayOffsets[5] = ras.readInt();
        overlayOffsets[6] = ras.readInt();
        overlayOffsets[7] = ras.readInt();
        overlayOffsets[8] = ras.readInt();
        for (i = 0; i < overlayOffsets.length; ++i) {
            this.parseOverlays(series, overlayOffsets[i], overlayKeys[i], store);
        }
        this.addMeta(keyPrefix + "ToolbarFlags", ras.readInt());
        ras.close();
        this.addMeta(keyPrefix + "DimensionZ", this.getSizeZ());
        this.addMeta(keyPrefix + "DimensionChannels", this.getSizeC());
        if (channelColorsOffset != 0L) {
            this.in.seek(channelColorsOffset + 16L);
            int namesOffset = this.in.readInt();
            if (namesOffset > 0) {
                this.in.skipBytes(namesOffset - 16);
                for (int i3 = 0; i3 < this.getSizeC() && this.in.getFilePointer() < this.in.length() - 1L; ++i3) {
                    String name = this.in.readCString();
                    if (name.length() > 128) continue;
                    this.addMeta(keyPrefix + "ChannelName" + i3, name);
                }
            }
        }
        if (timeStampOffset != 0L) {
            this.in.seek(timeStampOffset + 8L);
            for (i = 0; i < this.getSizeT(); ++i) {
                double stamp = this.in.readDouble();
                this.addMeta(keyPrefix + "TimeStamp" + i, stamp);
                this.timestamps.add(new Double(stamp));
            }
        }
        if (eventListOffset != 0L) {
            this.in.seek(eventListOffset + 4L);
            int numEvents = this.in.readInt();
            this.in.seek(this.in.getFilePointer() - 4L);
            this.in.order(!this.in.isLittleEndian());
            int tmpEvents = this.in.readInt();
            numEvents = numEvents < 0 ? tmpEvents : Math.min(numEvents, tmpEvents);
            this.in.order(!this.in.isLittleEndian());
            if (numEvents > 65535) {
                numEvents = 0;
            }
            for (int i4 = 0; i4 < numEvents; ++i4) {
                if (this.in.getFilePointer() + 16L > this.in.length()) continue;
                int size = this.in.readInt();
                double eventTime = this.in.readDouble();
                int eventType = this.in.readInt();
                this.addMeta(keyPrefix + "Event" + i4 + " Time", eventTime);
                this.addMeta(keyPrefix + "Event" + i4 + " Type", eventType);
                long fp = this.in.getFilePointer();
                int len = size - 16;
                if (len > 65536) {
                    len = 65536;
                }
                if (len < 0) {
                    len = 0;
                }
                this.addMeta(keyPrefix + "Event" + i4 + " Description", this.in.readString(len));
                this.in.seek(fp + (long)size - 16L);
                if (this.in.getFilePointer() < 0L) break;
            }
        }
        Hashtable<Integer, Integer> acquireChannels = new Hashtable<Integer, Integer>();
        Hashtable<String, Object> channelData = new Hashtable<String, Object>();
        Hashtable<Integer, Integer> acquireLasers = new Hashtable<Integer, Integer>();
        Hashtable<String, String> laserData = new Hashtable<String, String>();
        if (scanInformationOffset != 0L) {
            this.in.seek(scanInformationOffset);
            Stack<String> prefix = new Stack<String>();
            int count = 1;
            Object value = null;
            boolean done = false;
            int nextLaserMedium = 0;
            int nextLaserType = 0;
            boolean nextGain = false;
            boolean nextPinhole = false;
            boolean nextEmWave = false;
            boolean nextExWave = false;
            boolean nextChannelName = false;
            while (!done) {
                String v;
                int entry = this.in.readInt();
                int blockType = this.in.readInt();
                int dataSize = this.in.readInt();
                switch (blockType) {
                    case 0: {
                        switch (entry) {
                            case 0x10000000: {
                                prefix.push("Recording");
                                break;
                            }
                            case 0x30000000: {
                                prefix.push("Lasers");
                                break;
                            }
                            case 0x50000000: {
                                prefix.push("Laser " + count);
                                ++count;
                                break;
                            }
                            case 0x20000000: {
                                prefix.push("Tracks");
                                break;
                            }
                            case 0x40000000: {
                                prefix.push("Track " + count);
                                ++count;
                                break;
                            }
                            case 0x60000000: {
                                prefix.push("Detection Channels");
                                break;
                            }
                            case 0x70000000: {
                                prefix.push("Detection Channel " + count);
                                this.validChannels = ++count;
                                break;
                            }
                            case -2147483648: {
                                prefix.push("Illumination Channels");
                                break;
                            }
                            case -1879048192: {
                                prefix.push("Illumination Channel " + count);
                                ++count;
                                break;
                            }
                            case -1610612736: {
                                prefix.push("Beam Splitters");
                                break;
                            }
                            case -1342177280: {
                                prefix.push("Beam Splitter " + count);
                                ++count;
                                break;
                            }
                            case -1073741824: {
                                prefix.push("Data Channels");
                                break;
                            }
                            case -805306368: {
                                prefix.push("Data Channel " + count);
                                ++count;
                                break;
                            }
                            case 0x11000000: {
                                prefix.push("Timers");
                                break;
                            }
                            case 0x12000000: {
                                prefix.push("Timer " + count);
                                ++count;
                                break;
                            }
                            case 0x13000000: {
                                prefix.push("Markers");
                                break;
                            }
                            case 0x14000000: {
                                prefix.push("Marker " + count);
                                ++count;
                                break;
                            }
                            case -1: {
                                String p;
                                if (prefix.size() > 0 && (p = (String)prefix.pop()).endsWith("s")) {
                                    count = 1;
                                }
                                if (prefix.size() != 0) break;
                                done = true;
                            }
                        }
                        break;
                    }
                    case 4: {
                        value = new Long(this.in.readInt());
                        break;
                    }
                    case 5: {
                        value = new Double(this.in.readDouble());
                        break;
                    }
                    case 2: {
                        value = this.in.readString(dataSize);
                    }
                }
                String key = this.getKey(prefix, entry).trim();
                if (key != null) {
                    this.addMeta(keyPrefix + key, value);
                }
                String string = v = value != null ? value.toString().trim() : "";
                if (key.endsWith("Acquire")) {
                    Integer index = new Integer(count - 2);
                    Integer acquire = new Integer(v);
                    if (key.indexOf("Detection Channel") != -1) {
                        acquireChannels.put(index, acquire);
                    } else if (key.indexOf("Laser") != -1) {
                        acquireLasers.put(index, acquire);
                    }
                }
                switch (entry) {
                    case 0x10000002: {
                        store.setImageDescription(v, series);
                        break;
                    }
                    case 0x10000004: {
                        int next;
                        String[] tokens = v.split(" ");
                        StringBuffer model = new StringBuffer();
                        for (next = 0; next < tokens.length && tokens[next].indexOf("/") == -1; ++next) {
                            model.append(tokens[next]);
                        }
                        store.setObjectiveModel(model.toString(), series, 0);
                        if (next < tokens.length) {
                            String p = tokens[next++];
                            String mag = p.substring(0, p.indexOf("/") - 1);
                            String na = p.substring(p.indexOf("/") + 1);
                            try {
                                store.setObjectiveNominalMagnification(new Integer(mag), series, 0);
                            }
                            catch (NumberFormatException e) {
                                // empty catch block
                            }
                            try {
                                store.setObjectiveLensNA(new Float(na), series, 0);
                            }
                            catch (NumberFormatException e) {
                                // empty catch block
                            }
                        }
                        if (next < tokens.length) {
                            store.setObjectiveImmersion(tokens[next++], series, 0);
                        } else {
                            store.setObjectiveImmersion("Unknown", series, 0);
                        }
                        boolean iris = false;
                        if (next < tokens.length) {
                            iris = tokens[next++].trim().equalsIgnoreCase("iris");
                        }
                        store.setObjectiveIris(new Boolean(iris), series, 0);
                        store.setObjectiveCorrection("Unknown", series, 0);
                        store.setObjectiveID("Objective:" + series, series, 0);
                        store.setObjectiveSettingsObjective("Objective:" + series, series);
                        break;
                    }
                    case 0x4000000B: {
                        try {
                            store.setDimensionsTimeIncrement(new Float(v), series, 0);
                        }
                        catch (NumberFormatException e) {}
                        break;
                    }
                    case 0x50000001: {
                        String medium = v;
                        String laserType = "";
                        if (medium.startsWith("HeNe")) {
                            medium = "HeNe";
                            laserType = "Gas";
                        } else if (medium.startsWith("Argon")) {
                            medium = "Ar";
                            laserType = "Gas";
                        } else if (medium.equals("Titanium:Sapphire") || medium.equals("Mai Tai")) {
                            medium = "TiSapphire";
                            laserType = "SolidState";
                        } else if (medium.equals("YAG")) {
                            medium = "";
                            laserType = "SolidState";
                        } else if (medium.equals("Ar/Kr")) {
                            medium = "";
                            laserType = "Gas";
                        } else if (medium.equals("Enterprise")) {
                            medium = "";
                        }
                        laserData.put("medium " + nextLaserMedium, medium);
                        laserData.put("type " + nextLaserType, laserType);
                        ++nextLaserMedium;
                        ++nextLaserType;
                        break;
                    }
                    case 0x70000003: {
                        break;
                    }
                    case 0x70000009: {
                        try {
                            channelData.put("pinhole " + count, new Float(v));
                        }
                        catch (NumberFormatException e) {}
                        break;
                    }
                    case -1879048189: {
                        try {
                            Integer wave = new Integer((int)Float.parseFloat(v));
                            channelData.put("em " + count, wave);
                            channelData.put("ex " + count, wave);
                        }
                        catch (NumberFormatException e) {}
                        break;
                    }
                    case 268435510: {
                        double time = Double.parseDouble(v);
                        store.setImageCreationDate(DataTools.convertDate((long)(time * 8.64E7), 2), series);
                        break;
                    }
                    case -805306367: {
                        channelData.put("name " + count, v);
                    }
                }
                if (done) continue;
                done = this.in.getFilePointer() >= this.in.length() - 12L;
            }
        }
        int nextLaser = 0;
        for (int i5 = 0; i5 < acquireLasers.size(); ++i5) {
            boolean acquire;
            boolean bl2 = acquire = (Integer)acquireLasers.get(new Integer(i5)) != 0;
            if (!acquire) continue;
            String medium = (String)laserData.get("medium " + i5);
            String laserType = (String)laserData.get("type " + i5);
            if (medium.equals("") || laserType.equals("")) continue;
            store.setLaserLaserMedium(medium, 0, nextLaser);
            store.setLaserType(laserType, 0, nextLaser);
            if (nextLaser < this.getEffectiveSizeC()) {
                store.setLightSourceID("LightSource:" + nextLaser, 0, nextLaser);
                store.setLightSourceSettingsLightSource("LightSource:" + nextLaser, series, nextLaser);
            }
            ++nextLaser;
        }
        int nextChannel = 0;
        for (int i6 = 0; i6 < acquireChannels.size(); ++i6) {
            boolean acquire;
            boolean bl3 = acquire = (Integer)acquireChannels.get(new Integer(i6 + 1)) != 0;
            if (!acquire) continue;
            String name = (String)channelData.get("name " + (i6 + 3));
            Integer ex = (Integer)channelData.get("ex " + (i6 + 3));
            Integer em = (Integer)channelData.get("em " + (i6 + 3));
            Float pinhole = (Float)channelData.get("pinhole " + (i6 + 3));
            store.setLogicalChannelName(name, series, nextChannel);
            store.setLogicalChannelEmWave(em, series, nextChannel);
            store.setLogicalChannelExWave(ex, series, nextChannel);
            store.setLogicalChannelPinholeSize(pinhole, series, nextChannel);
            ++nextChannel;
        }
        Float pixX = new Float((float)this.pixelSizeX);
        Float pixY = new Float((float)this.pixelSizeY);
        Float pixZ = new Float((float)this.pixelSizeZ);
        store.setDimensionsPhysicalSizeX(pixX, series, 0);
        store.setDimensionsPhysicalSizeY(pixY, series, 0);
        store.setDimensionsPhysicalSizeZ(pixZ, series, 0);
        int offset = 0;
        for (int i7 = 0; i7 < series; ++i7) {
            this.setSeries(i7);
            offset += this.getImageCount();
        }
        this.setSeries(series);
        float firstStamp = this.timestamps.size() <= offset ? 0.0f : ((Double)this.timestamps.get(offset)).floatValue();
        for (int i8 = 0; i8 < this.getImageCount(); ++i8) {
            float nextStamp;
            int[] zct = FormatTools.getZCTCoords(this, i8);
            if (offset + zct[2] >= this.timestamps.size()) continue;
            float thisStamp = ((Double)this.timestamps.get(offset + zct[2])).floatValue();
            store.setPlaneTimingDeltaT(new Float(thisStamp - firstStamp), series, 0, i8);
            float f = nextStamp = zct[2] < this.getSizeT() - 1 ? ((Double)this.timestamps.get(offset + zct[2] + 1)).floatValue() : thisStamp;
            if (i8 == this.getSizeT() - 1 && zct[2] > 0) {
                thisStamp = ((Double)this.timestamps.get(offset + zct[2] - 1)).floatValue();
            }
            store.setPlaneTimingExposureTime(new Float(nextStamp - thisStamp), series, 0, i8);
        }
        this.in.close();
    }

    protected void parseOverlays(int series, long data, String suffix, MetadataStore store) throws IOException {
        if (data == 0L) {
            return;
        }
        String prefix = "Series " + series + " ";
        this.in.seek(data);
        int numberOfShapes = this.in.readInt();
        int size = this.in.readInt();
        if (size <= 194) {
            return;
        }
        this.in.skipBytes(20);
        boolean valid = this.in.readInt() == 1;
        this.in.skipBytes(164);
        for (int i = 0; i < numberOfShapes; ++i) {
            long offset = this.in.getFilePointer();
            int type = this.in.readInt();
            int blockLength = this.in.readInt();
            int lineWidth = this.in.readInt();
            int measurements = this.in.readInt();
            double textOffsetX = this.in.readDouble();
            double textOffsetY = this.in.readDouble();
            int color = this.in.readInt();
            boolean validShape = this.in.readInt() != 0;
            int knotWidth = this.in.readInt();
            int catchArea = this.in.readInt();
            int fontHeight = this.in.readInt();
            int fontWidth = this.in.readInt();
            int fontEscapement = this.in.readInt();
            int fontOrientation = this.in.readInt();
            int fontWeight = this.in.readInt();
            boolean fontItalic = this.in.readInt() != 0;
            boolean fontUnderlined = this.in.readInt() != 0;
            boolean fontStrikeout = this.in.readInt() != 0;
            int fontCharSet = this.in.readInt();
            int fontOutputPrecision = this.in.readInt();
            int fontClipPrecision = this.in.readInt();
            int fontQuality = this.in.readInt();
            int fontPitchAndFamily = this.in.readInt();
            String fontName = DataTools.stripString(this.in.readString(64));
            boolean enabled = this.in.readShort() == 0;
            boolean moveable = this.in.readInt() == 0;
            this.in.skipBytes(34);
            store.setShapeFontFamily(fontName, series, 0, i);
            store.setShapeFontSize(new Integer(fontHeight), series, 0, i);
            store.setShapeFontStyle(fontItalic ? "normal" : "italic", series, 0, i);
            store.setShapeFontWeight(String.valueOf(fontWeight), series, 0, i);
            store.setShapeLocked(new Boolean(moveable), series, 0, i);
            store.setShapeStrokeColor(String.valueOf(color), series, 0, i);
            store.setShapeStrokeWidth(new Integer(lineWidth), series, 0, i);
            store.setShapeTextDecoration(fontUnderlined ? "underline" : (fontStrikeout ? "line-through" : "normal"), series, 0, i);
            store.setShapeVisibility(new Boolean(enabled), series, 0, i);
            switch (type) {
                case 13: {
                    double x = this.in.readDouble();
                    double y = this.in.readDouble();
                    String text = DataTools.stripString(this.in.readCString());
                    store.setShapeText(text, series, 0, i);
                    break;
                }
                case 14: {
                    this.in.skipBytes(4);
                    double startX = this.in.readDouble();
                    double startY = this.in.readDouble();
                    double endX = this.in.readDouble();
                    double endY = this.in.readDouble();
                    store.setLineX1(String.valueOf(startX), series, 0, i);
                    store.setLineY1(String.valueOf(startY), series, 0, i);
                    store.setLineX2(String.valueOf(endX), series, 0, i);
                    store.setLineY2(String.valueOf(endY), series, 0, i);
                    break;
                }
                case 15: 
                case 16: 
                case 17: 
                case 25: {
                    this.in.skipBytes(36);
                    break;
                }
                case 18: {
                    this.in.skipBytes(4);
                    double topX = this.in.readDouble();
                    double topY = this.in.readDouble();
                    double bottomX = this.in.readDouble();
                    double bottomY = this.in.readDouble();
                    double width = Math.abs(bottomX - topX);
                    double height = Math.abs(bottomY - topY);
                    topX = Math.min(topX, bottomX);
                    topY = Math.min(topY, bottomY);
                    store.setRectX(String.valueOf(topX), series, 0, i);
                    store.setRectY(String.valueOf(topY), series, 0, i);
                    store.setRectWidth(String.valueOf(width), series, 0, i);
                    store.setRectHeight(String.valueOf(height), series, 0, i);
                    break;
                }
                case 19: {
                    int knots = this.in.readInt();
                    double[] xs = new double[knots];
                    double[] ys = new double[knots];
                    for (int j = 0; j < xs.length; ++j) {
                        xs[j] = this.in.readDouble();
                        ys[j] = this.in.readDouble();
                    }
                    double rx = 0.0;
                    double ry = 0.0;
                    double centerX = 0.0;
                    double centerY = 0.0;
                    if (knots == 4) {
                        double r1x = Math.abs(xs[2] - xs[0]) / 2.0;
                        double r1y = Math.abs(ys[2] - ys[0]) / 2.0;
                        double r2x = Math.abs(xs[3] - xs[1]) / 2.0;
                        double r2y = Math.abs(ys[3] - ys[1]) / 2.0;
                        if (r1x > r2x) {
                            ry = r1y;
                            rx = r2x;
                            centerX = Math.min(xs[3], xs[1]) + rx;
                            centerY = Math.min(ys[2], ys[0]) + ry;
                        } else {
                            ry = r2y;
                            rx = r1x;
                            centerX = Math.min(xs[2], xs[0]) + rx;
                            centerY = Math.min(ys[3], ys[1]) + ry;
                        }
                    } else if (knots == 3) {
                        centerX = xs[0];
                        centerY = ys[0];
                        rx = Math.sqrt(Math.pow(xs[1] - xs[0], 2.0) + Math.pow(ys[1] - ys[0], 2.0));
                        ry = Math.sqrt(Math.pow(xs[2] - xs[0], 2.0) + Math.pow(ys[2] - ys[0], 2.0));
                        double slope = (ys[2] - centerY) / (xs[2] - centerX);
                        double theta = Math.toDegrees(Math.atan(slope));
                        store.setEllipseTransform("rotate(" + theta + " " + centerX + " " + centerY + ")", series, 0, i);
                    }
                    store.setEllipseCx(String.valueOf(centerX), series, 0, i);
                    store.setEllipseCy(String.valueOf(centerY), series, 0, i);
                    store.setEllipseRx(String.valueOf(rx), series, 0, i);
                    store.setEllipseRy(String.valueOf(ry), series, 0, i);
                    break;
                }
                case 24: {
                    this.in.skipBytes(4);
                    double centerX = this.in.readDouble();
                    double centerY = this.in.readDouble();
                    double curveX = this.in.readDouble();
                    double curveY = this.in.readDouble();
                    double radius = Math.sqrt(Math.pow(curveX - centerX, 2.0) + Math.pow(curveY - centerY, 2.0));
                    store.setCircleCx(String.valueOf(centerX), series, 0, i);
                    store.setCircleCy(String.valueOf(centerY), series, 0, i);
                    store.setCircleR(String.valueOf(radius), series, 0, i);
                    break;
                }
                case 29: {
                    this.in.skipBytes(4);
                    double[][] points = new double[3][2];
                    for (int j = 0; j < points.length; ++j) {
                        for (int k = 0; k < points[j].length; ++k) {
                            points[j][k] = this.in.readDouble();
                        }
                    }
                    double s = 0.5 * ((points[1][0] - points[2][0]) * (points[0][0] - points[2][0]) - (points[1][1] - points[2][1]) * (points[2][1] - points[0][1]));
                    double div = (points[0][0] - points[1][0]) * (points[2][1] - points[0][1]) - (points[1][1] - points[0][1]) * (points[0][0] - points[2][0]);
                    double cx = 0.5 * (points[0][0] + points[1][0]) + (s /= div) * (points[1][1] - points[0][1]);
                    double cy = 0.5 * (points[0][1] + points[1][1]) + s * (points[0][0] - points[1][0]);
                    double r = Math.sqrt(Math.pow(points[0][0] - cx, 2.0) + Math.pow(points[0][1] - cy, 2.0));
                    store.setCircleCx(String.valueOf(cx), series, 0, i);
                    store.setCircleCy(String.valueOf(cy), series, 0, i);
                    store.setCircleR(String.valueOf(r), series, 0, i);
                    break;
                }
                case 28: {
                    this.in.skipBytes(4);
                    double[][] points = new double[3][2];
                    for (int j = 0; j < points.length; ++j) {
                        for (int k = 0; k < points[j].length; ++k) {
                            points[j][k] = this.in.readDouble();
                        }
                    }
                    StringBuffer p = new StringBuffer();
                    for (int j = 0; j < points.length; ++j) {
                        p.append(points[j][0]);
                        p.append(",");
                        p.append(points[j][1]);
                        if (j >= points.length - 1) continue;
                        p.append(" ");
                    }
                    store.setPolylinePoints(p.toString(), series, 0, i);
                    break;
                }
                case 20: 
                case 21: 
                case 26: {
                    int k;
                    int j;
                    int nKnots = this.in.readInt();
                    double[][] points = new double[nKnots][2];
                    for (j = 0; j < points.length; ++j) {
                        for (k = 0; k < points[j].length; ++k) {
                            points[j][k] = this.in.readDouble();
                        }
                    }
                    StringBuffer p = new StringBuffer();
                    for (j = 0; j < points.length; ++j) {
                        p.append(points[j][0]);
                        p.append(",");
                        p.append(points[j][1]);
                        if (j >= points.length - 1) continue;
                        p.append(" ");
                    }
                    if (type == 20) {
                        store.setPolygonPoints(p.toString(), series, 0, i);
                        break;
                    }
                    store.setPolylinePoints(p.toString(), series, 0, i);
                    break;
                }
                case 22: 
                case 23: 
                case 27: {
                    int k;
                    int j;
                    int nKnots = this.in.readInt();
                    double[][] points = new double[nKnots][2];
                    for (j = 0; j < points.length; ++j) {
                        for (k = 0; k < points[j].length; ++k) {
                            points[j][k] = this.in.readDouble();
                        }
                    }
                    break;
                }
            }
            this.in.seek(offset + (long)blockLength);
        }
    }

    private String getKey(Stack stack, int entry) {
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < stack.size(); ++i) {
            sb.append((String)stack.get(i));
            sb.append("/");
        }
        sb.append(" - ");
        sb.append(metadataKeys.get(new Integer(entry)));
        return sb.toString();
    }

    private String[] parseMDB(String mdbFile) throws FormatException {
        Location mdb = new Location(mdbFile).getAbsoluteFile();
        Location parent = mdb.getParentFile();
        Vector<String[]>[] tables = MDBParser.parseDatabase(mdbFile);
        Vector<String> referencedLSMs = new Vector<String>();
        for (int table = 0; table < tables.length; ++table) {
            String[] columnNames = tables[table].get(0);
            String tableName = columnNames[0];
            for (int row = 1; row < tables[table].size(); ++row) {
                String[] tableRow = tables[table].get(row);
                for (int col = 0; col < tableRow.length; ++col) {
                    Location file2;
                    String key = tableName + " " + columnNames[col + 1] + " " + row;
                    this.addMeta(key, tableRow[col]);
                    if (!tableName.equals("Recordings") || columnNames[col + 1] == null || !columnNames[col + 1].equals("SampleData")) continue;
                    String filename = tableRow[col].trim();
                    filename = filename.replace('\\', File.separatorChar);
                    filename = filename.replaceAll("/", File.separator);
                    if ((filename = filename.substring(filename.lastIndexOf(File.separator) + 1)).length() <= 0 || !(file2 = new Location(parent, filename)).exists()) continue;
                    referencedLSMs.add(file2.getAbsolutePath());
                }
            }
        }
        if (referencedLSMs.size() > 0) {
            return referencedLSMs.toArray(new String[0]);
        }
        String[] fileList = parent.list();
        for (int i = 0; i < fileList.length; ++i) {
            if (!ZeissLSMReader.checkSuffix(fileList[i], new String[]{"lsm"})) continue;
            referencedLSMs.add(new Location(parent, fileList[i]).getAbsolutePath());
        }
        return referencedLSMs.toArray(new String[0]);
    }

    private static Hashtable createKeys() {
        Hashtable<Integer, String> h = new Hashtable<Integer, String>();
        h.put(new Integer(0x10000001), "Name");
        h.put(new Integer(0x4000000C), "Name");
        h.put(new Integer(0x50000001), "Name");
        h.put(new Integer(-1879048191), "Name");
        h.put(new Integer(-1879048187), "Name");
        h.put(new Integer(-1342177277), "Name");
        h.put(new Integer(-805306367), "Name");
        h.put(new Integer(0x12000001), "Name");
        h.put(new Integer(0x14000001), "Name");
        h.put(new Integer(0x10000002), "Description");
        h.put(new Integer(335544322), "Description");
        h.put(new Integer(0x10000003), "Notes");
        h.put(new Integer(0x10000004), "Objective");
        h.put(new Integer(0x10000005), "Processing Summary");
        h.put(new Integer(0x10000006), "Special Scan Mode");
        h.put(new Integer(0x10000007), "Scan Type");
        h.put(new Integer(0x10000008), "Scan Mode");
        h.put(new Integer(0x10000009), "Number of Stacks");
        h.put(new Integer(0x1000000A), "Lines Per Plane");
        h.put(new Integer(0x1000000B), "Samples Per Line");
        h.put(new Integer(0x1000000C), "Planes Per Volume");
        h.put(new Integer(0x1000000D), "Images Width");
        h.put(new Integer(0x1000000E), "Images Height");
        h.put(new Integer(0x1000000F), "Number of Planes");
        h.put(new Integer(0x10000010), "Number of Stacks");
        h.put(new Integer(0x10000011), "Number of Channels");
        h.put(new Integer(0x10000012), "Linescan XY Size");
        h.put(new Integer(0x10000013), "Scan Direction");
        h.put(new Integer(0x10000014), "Time Series");
        h.put(new Integer(0x10000015), "Original Scan Data");
        h.put(new Integer(0x10000016), "Zoom X");
        h.put(new Integer(0x10000017), "Zoom Y");
        h.put(new Integer(0x10000018), "Zoom Z");
        h.put(new Integer(0x10000019), "Sample 0X");
        h.put(new Integer(0x1000001A), "Sample 0Y");
        h.put(new Integer(0x1000001B), "Sample 0Z");
        h.put(new Integer(0x1000001C), "Sample Spacing");
        h.put(new Integer(0x1000001D), "Line Spacing");
        h.put(new Integer(0x1000001E), "Plane Spacing");
        h.put(new Integer(0x1000001F), "Plane Width");
        h.put(new Integer(0x10000020), "Plane Height");
        h.put(new Integer(0x10000021), "Volume Depth");
        h.put(new Integer(268435508), "Rotation");
        h.put(new Integer(268435509), "Precession");
        h.put(new Integer(268435510), "Sample 0Time");
        h.put(new Integer(268435511), "Start Scan Trigger In");
        h.put(new Integer(268435512), "Start Scan Trigger Out");
        h.put(new Integer(268435513), "Start Scan Event");
        h.put(new Integer(0x10000040), "Start Scan Time");
        h.put(new Integer(0x10000041), "Stop Scan Trigger In");
        h.put(new Integer(268435522), "Stop Scan Trigger Out");
        h.put(new Integer(268435523), "Stop Scan Event");
        h.put(new Integer(0x10000044), "Stop Scan Time");
        h.put(new Integer(268435525), "Use ROIs");
        h.put(new Integer(268435526), "Use Reduced Memory ROIs");
        h.put(new Integer(268435527), "User");
        h.put(new Integer(268435528), "Use B/C Correction");
        h.put(new Integer(268435529), "Position B/C Contrast 1");
        h.put(new Integer(0x10000050), "Position B/C Contrast 2");
        h.put(new Integer(0x10000051), "Interpolation Y");
        h.put(new Integer(268435538), "Camera Binning");
        h.put(new Integer(268435539), "Camera Supersampling");
        h.put(new Integer(268435540), "Camera Frame Width");
        h.put(new Integer(0x10000055), "Camera Frame Height");
        h.put(new Integer(268435542), "Camera Offset X");
        h.put(new Integer(268435543), "Camera Offset Y");
        h.put(new Integer(0x40000001), "Multiplex Type");
        h.put(new Integer(0x40000002), "Multiplex Order");
        h.put(new Integer(0x40000003), "Sampling Mode");
        h.put(new Integer(0x40000004), "Sampling Method");
        h.put(new Integer(0x40000005), "Sampling Number");
        h.put(new Integer(0x40000006), "Acquire");
        h.put(new Integer(0x50000002), "Acquire");
        h.put(new Integer(0x7000000B), "Acquire");
        h.put(new Integer(-1879048188), "Acquire");
        h.put(new Integer(-805306345), "Acquire");
        h.put(new Integer(0x40000007), "Sample Observation Time");
        h.put(new Integer(0x40000008), "Time Between Stacks");
        h.put(new Integer(0x4000000D), "Collimator 1 Name");
        h.put(new Integer(0x4000000E), "Collimator 1 Position");
        h.put(new Integer(0x4000000F), "Collimator 2 Name");
        h.put(new Integer(0x40000010), "Collimator 2 Position");
        h.put(new Integer(0x40000011), "Is Bleach Track");
        h.put(new Integer(1073741842), "Bleach After Scan Number");
        h.put(new Integer(1073741843), "Bleach Scan Number");
        h.put(new Integer(0x40000014), "Trigger In");
        h.put(new Integer(301989892), "Trigger In");
        h.put(new Integer(335544323), "Trigger In");
        h.put(new Integer(1073741845), "Trigger Out");
        h.put(new Integer(301989893), "Trigger Out");
        h.put(new Integer(0x14000004), "Trigger Out");
        h.put(new Integer(1073741846), "Is Ratio Track");
        h.put(new Integer(1073741847), "Bleach Count");
        h.put(new Integer(1073741848), "SPI Center Wavelength");
        h.put(new Integer(1073741849), "Pixel Time");
        h.put(new Integer(0x40000020), "ID Condensor Frontlens");
        h.put(new Integer(1073741857), "Condensor Frontlens");
        h.put(new Integer(0x40000022), "ID Field Stop");
        h.put(new Integer(1073741859), "Field Stop Value");
        h.put(new Integer(0x40000024), "ID Condensor Aperture");
        h.put(new Integer(1073741861), "Condensor Aperture");
        h.put(new Integer(1073741862), "ID Condensor Revolver");
        h.put(new Integer(1073741863), "Condensor Revolver");
        h.put(new Integer(1073741864), "ID Transmission Filter 1");
        h.put(new Integer(1073741865), "ID Transmission 1");
        h.put(new Integer(0x40000030), "ID Transmission Filter 2");
        h.put(new Integer(1073741873), "ID Transmission 2");
        h.put(new Integer(1073741874), "Repeat Bleach");
        h.put(new Integer(0x40000033), "Enable Spot Bleach Pos");
        h.put(new Integer(0x40000034), "Spot Bleach Position X");
        h.put(new Integer(1073741877), "Spot Bleach Position Y");
        h.put(new Integer(1073741878), "Bleach Position Z");
        h.put(new Integer(0x50000003), "Power");
        h.put(new Integer(-1879048190), "Power");
        h.put(new Integer(0x70000003), "Detector Gain");
        h.put(new Integer(0x70000005), "Amplifier Gain");
        h.put(new Integer(0x70000007), "Amplifier Offset");
        h.put(new Integer(0x70000009), "Pinhole Diameter");
        h.put(new Integer(0x7000000C), "Detector Name");
        h.put(new Integer(0x7000000D), "Amplifier Name");
        h.put(new Integer(0x7000000E), "Pinhole Name");
        h.put(new Integer(0x7000000F), "Filter Set Name");
        h.put(new Integer(0x70000010), "Filter Name");
        h.put(new Integer(1879048211), "Integrator Name");
        h.put(new Integer(1879048212), "Detection Channel Name");
        h.put(new Integer(1879048213), "Detector Gain B/C 1");
        h.put(new Integer(1879048214), "Detector Gain B/C 2");
        h.put(new Integer(0x70000017), "Amplifier Gain B/C 1");
        h.put(new Integer(1879048216), "Amplifier Gain B/C 2");
        h.put(new Integer(1879048217), "Amplifier Offset B/C 1");
        h.put(new Integer(0x70000020), "Amplifier Offset B/C 2");
        h.put(new Integer(1879048225), "Spectral Scan Channels");
        h.put(new Integer(0x70000022), "SPI Wavelength Start");
        h.put(new Integer(1879048227), "SPI Wavelength End");
        h.put(new Integer(1879048230), "Dye Name");
        h.put(new Integer(-805306348), "Dye Name");
        h.put(new Integer(0x70000027), "Dye Folder");
        h.put(new Integer(-805306347), "Dye Folder");
        h.put(new Integer(-1879048189), "Wavelength");
        h.put(new Integer(-1879048186), "Power B/C 1");
        h.put(new Integer(-1879048185), "Power B/C 2");
        h.put(new Integer(-1342177279), "Filter Set");
        h.put(new Integer(-1342177278), "Filter");
        h.put(new Integer(-805306364), "Color");
        h.put(new Integer(-805306363), "Sample Type");
        h.put(new Integer(-805306362), "Bits Per Sample");
        h.put(new Integer(-805306361), "Ratio Type");
        h.put(new Integer(-805306360), "Ratio Track 1");
        h.put(new Integer(-805306359), "Ratio Track 2");
        h.put(new Integer(-805306358), "Ratio Channel 1");
        h.put(new Integer(-805306357), "Ratio Channel 2");
        h.put(new Integer(-805306356), "Ratio Const. 1");
        h.put(new Integer(-805306355), "Ratio Const. 2");
        h.put(new Integer(-805306354), "Ratio Const. 3");
        h.put(new Integer(-805306353), "Ratio Const. 4");
        h.put(new Integer(-805306352), "Ratio Const. 5");
        h.put(new Integer(-805306351), "Ratio Const. 6");
        h.put(new Integer(-805306350), "Ratio First Images 1");
        h.put(new Integer(-805306349), "Ratio First Images 2");
        h.put(new Integer(-805306346), "Spectrum");
        h.put(new Integer(301989891), "Interval");
        return h;
    }
}

