function aamlib_scaleImagesAndPointModelsInProject
% function aamlib_scaleImagesAndPointModelsInProject
%
% This function allows the user to rescale the images and the point models
% for a particular project. This function is not available from the GUI for
% safety reasons. To explain the reason for this method, there is an
% example of how to use it below:
%
% Suppose we have two projects PRJ_080516_All_species_FP-2-Max10 (images at
% 60pix/cm) and PRJ_species FP changed (images at 240pix/cm). To make the
% two projects comparable or to merge the two projects we need to scale the
% images and point models in the first project by a factor of 4. To do this
% we make a new project called PRJ_080516_All_species_FP-2-Max10_240pxpcm (images at 240pix/cm) 
% where pxpcm denotes pix/cm. 
%
% To make this new project we first copied the original project 
%
% PRJ_080516_All_species_FP-2-Max10
%
% using explorer and renamed it to
%
% PRJ_080516_All_species_FP-2-Max10_240pxpcm
%
% then in Matlab we changed directory to
%
% PRJ_080516_All_species_FP-2-Max10_240pxpcm 
%
% and at the Matlab prompt typed:
%
% >>aamlib_scaleImagesAndPointModelsInProject
%
% this changed the size of the images and the point models in project.
% NOTE: THIS DID NOT CHANGE THE TEMPLATE AT ALL
%
% Dr. A. I. Hanna, CMP, 2008

if showWarning()==0
    fprintf('User cancelled operation\n');
    return;
end
scalingfactor = getScaleFactor;
if (scalingfactor<=0)
    uiwait(msgbox('Invalid scaling factor','Error','modal'));
    fprintf('Systen cancelled operation\n');
    return;
end
if (scalingfactor==1)
    uiwait(msgbox(sprintf('Scaling factor is 1.\nNo scaling required'),'Report','modal'));
    fprintf('Systen cancelled operation\n');
    return;
end
fprintf(' - scaling factor = %f\n', scalingfactor);
[imgfiles, imgdir] = getImageNames;
if isempty(imgfiles)
    str = sprintf('Could not find any image files in directory Cropped.\n Please check you are in a valid project');
    uiwait(msgbox(str,'No Images','modal'));
    return;
end
scaleAllImagesAndPointModels(imgfiles, imgdir, scalingfactor);
return;
%%
function scaleAllImagesAndPointModels(imgfiles, imgdir, scalingfactor)
for i=1:length(imgfiles)
    imgfile = imgfiles{i};
    fprintf(' - scaling data for %s...', imgfile);
    scaleImageAndPointModels(imgfile, imgdir, scalingfactor);
    fprintf('done.\n');
end
return
%%
function scaleImageAndPointModels(imgfile, imgdir, scalingfactor)
fimgfile = [imgdir, filesep, imgfile];
if ~exist(fimgfile, 'file')
    return;
end
rescaleImage(fimgfile, scalingfactor);
rescalePointModels(fimgfile, scalingfactor);
return
%%
function rescalePointModels(fimgfile, scalingfactor)
pmfiles = findPMFiles(fimgfile);
dim = 2;
for i=1:length(pmfiles)
    pmfile = pmfiles{i};
    if exist(pmfile, 'file')
        pts = load(pmfile);
        if isfield(pts, 'pts')
            pts = pts.pts;
            pts = (pts)*scalingfactor;
            save(pmfile, 'pts');
        end
    end
end
return
%%
function rescaleImage(imgname, scalingfactor)
I = imread(imgname);
I = imresize(I, scalingfactor);
imwrite(I, imgname, 'JPG');
return
%%
function pmfiles = findPMFiles(imgname)
[junk, pmname] = fileparts(imgname);
pmname = [pmname, '_pm.mat'];
pmdir = [pwd, filesep, 'PointModels'];
pmfiles=FindFilenames(pmname, pmdir);
return
%%
function scalingfactor = getScaleFactor
prompt={'Enter the scaling factor for this project:'};
name='Scaling Factor';
numlines=1;
defaultanswer={'1'};
answer=inputdlg(prompt,name,numlines,defaultanswer);
scalingfactor = -1;
if isempty(answer)
    return;
end
scalingfactor = str2double(answer{1});
return
%%
function [files, imgdir] = getImageNames
imgdir = [pwd, filesep, 'Cropped'];
imgfilt = [imgdir, filesep, '*.jpg'];
files = dir(imgfilt);
files = {files.name};
return
%%
function cont = showWarning
l1 = 'This method will change the scale of BOTH the images and the point models';
l2 = 'in this project. If you have not already done so, it is STRONGLY advised';
l3 = 'that you copy your original data before continuing. You can cancel this operation by hitting Cancel.';

str = sprintf('%s\n%s\n%s', l1, l2, l3);
answer = questdlg(str,'Continue', 'Continue', 'Cancel', 'Cancel');
cont = 0;
if strcmpi(answer, 'continue')
    cont = 1;
end
return