function m = leaf_rasterize( m, varargin )
%m = leaf_rasterize( m, ... )
%   Convert a mesh to a 3D raster scan.  The raster is represented as a
%   series of PNG files, one for each section through the mesh.  The raster
%   scan will be coloured according to the current plot settings for the
%   mesh.
%
%   Options:
%       filename    The pattern of the names of the files to create.  This
%                   should contain the substring '%d', which will be
%                   replaced by the index number of each section through
%                   the mesh.  The name can be a relative or absolute path
%                   name.  If relative, it will be understood as relative
%                   to the 'dirname' argument, if present, otherwise
%                   relative to the project directory's 'meshes'
%                   subdirectory, or if the mesh does not belong to a
%                   project, the current working directory.
%       dirname     The direectory to save the files in.  Ignored if the
%                   'filename' argument was an absolute path name.
%       rastersize  A vector of three integers [X Y Z], specifying how many
%                   pixels the raster should be in each direction.  The
%                   mesh will be centered in the resulting 3D image, and
%                   will be uniformly scaled so as to fill the space, less
%                   whatever margin is specified.
%       margin      A single integer, a triple [X Y Z], or a sextuple [X1 X2
%                   Y1 Y2 Z1 Z2].  The specified number of pixels will be
%                   left blank on the faces of the raster and the mesh
%                   rendered within the remaining volume.

    if isempty(m), return; end
    
    [s,ok] = safemakestruct( mfilename(), varargin );
    if ~ok, return; end

    s = defaultfields( s,...
            'filename', '', ...
            'dirname', '', ...
            'rastersize', 512, ...
            'margin', 0 );
    ok = checkcommandargs( mfilename(), s, 'exact', ...
            'filename', 'dirname', 'rastersize', 'margin' );
    if ~ok, return; end

    if s.rastersize < 0
        complain( 'rastersize must be a positive integer, %f found.', ...
            s.rastersize );
        return;
    end
    
    margin = max( floor( s.margin(:)' ), 0 );
    if isempty(margin)
        margin = [0 0 0 0 0 0];
    elseif length(margin)==1
        margin = repmat( margin, 1, 6 );
    elseif length(margin)==3
        margin = reshape( [ margin; margin ], 1, [] );
    elseif length(margin) < 6
        complain( 'margin must have 1, 3, or 6 elements, %d found.', ...
            length(margin) );
        return;
    else
        margin = margin( 1:6 );
    end
    
    if ~isabsolutepath( s.filename )
        if isempty(s.dirname)
            if isempty(m.globalProps.projectdir)
                s.dirname = pwd;
            else
                s.dirname = fullfile( m.globalProps.projectdir, 'meshes' );
            end
        end
        fullfilename = fullfile( s.dirname, s.filename );
        [s.dirname, s.filename, s.ext] = fileparts( fullfilename );
        if isempty(regexp( s.filename, '%d' ))
            s.filename = regexprep( s.filename, '%', '%%' );
            s.filename = [ s.filename, '%d' ];
        end
    end
    s.ext = '.png';
    s.dirname
    s.filename
    s.ext
    
    marginlo = margin([1 3 5]);
    marginhi = margin([2 4 6]);
    innersize = rastersize - sum( marginlo, marginhi );
    
end

