function [s,savedrunname,savedrundesc] = MoveStagesToProject(ProjectDirectory)
%MoveStagesToProject(ProjectDirectory)
%
%Move 'stage' files resulting from a run conducted on a remote cluster
%machine into the project whose directory is ProjectDirectory, replacing
%any existing stage files there.
%
%The files to be copied are assumed to reside in directories of the form
%[ProjectDirectory]/runs/[experimentname]/meshes.  The folder
%[ProjectDirectory]/runs/[experimentname] is also assumed to contain a
%file called 'CommandLine.txt'.  The presence of this file is what tells
%this procedure that [ProjectDirectory]/runs/[experimentname] is an
%experimental results directory.
%
%If there is more than one such directory, the user is asked to choose one,
%each being identified to the user by the first line of its CommandLine.txt
%file.
%
%The result is empty in case of any problems, otherwise a positive number
%(whose value is not otherwise meaningful).  When there is a problem, a
%message will be printed to the Matlab command window.
%
%J.Andrew Bangham 2008

    moveOldRunsToNewRuns( ProjectDirectory );
    s = [];
    savedrunname = '';
    savedrundesc = '';
    if nargin<1
        ProjectDirectory = pwd;
    else
        ProjectDirectory = fullpath(ProjectDirectory,pwd);
    end

    RUNSDIR = 'runs';
    OLDRUNSDIR = 'movies';
    dirname = fullfile(ProjectDirectory,RUNSDIR);
    if ~exist( dirname, 'dir' )
        fprintf( 1, 'No runs directory in project %s\n', ProjectDirectory );
        return;
    end
    d = dir(dirname);
    experimentdir = {};
    commandlines = {};
    commandchars = [];
    for i = 1:length(d)
        name = d(i).name;
        if (~strcmp(name,'.')) && (~strcmp(name,'..'))
          % disp(name);
            expt = fullfile(dirname,name);
            commandFileName = 'CommandLine.txt';
            fh = fopen(fullfile(expt,commandFileName),'r');
            if fh ~= -1
                experimentdir{end+1} = expt;
                line = fgetl(fh);
                if iseof( line )
                  % fprintf( 1, 'No command line in %s in %s\n', commandFileName, dirname );
                    continue;
                end
                commandlines{end+1} = line;
                commandchars(end+1) = length(commandlines{end});
                fclose( fh );
            end
        end
    end
    if isempty(commandlines)
        fprintf( 1, 'No suitable files in directory %s\n', dirname );
        return;
    end
    commandsize = max(commandchars);
    commandnumber = length(commandchars);
    charwidth = 6;
    charheight = 18;
    s = listdlg( 'PromptString', 'Select an experiment:', ...
                     'SelectionMode', 'single', ...
                     'ListString', commandlines, ...
                     'ListSize', max( [600,200], [commandsize*charwidth,commandnumber*charheight] ) );
    if ~isempty(s)
        [pathstr,savedrunname] = fileparts( experimentdir{s} );
        savedrundesc = commandlines{s};
        exptdir = dir(fullfile(experimentdir{s},'meshes','*_s0*.mat'));
        if ~isempty(exptdir)
            % remove stages files from project directory
            delete(fullfile(ProjectDirectory,'*_s0*.mat'));
            % clear stage files JAB
            h_menu_item=findobj(0,'tag','deleteStagesAndTimesItem'); % find the menu item
            handles = guidata( h_menu_item );
            m=handles.mesh;
            m = leaf_deletestages( m, 'stages', true, 'times', true );
            % copy stages file from experiment into project directory
            for i = 1:length(exptdir)
                result = exptdir(i).name;
                copyfile(fullfile(experimentdir{s},'meshes',result),ProjectDirectory);
                % update list of stage files JAB
                ind=regexp(result,'_s0');
                time_string=result(ind+3:end-4);
                new_stage_time = stageTextToTime( time_string ); 
                m.stagetimes = addStages( m.stagetimes, new_stage_time );
            end
            % remake the GUI menu item JAB
            handles.mesh=m;
            handles = remakeStageMenu( handles );
        else
            fprintf( 1, 'No suitable files found in remote directory\n    %s\nOld stage files not deleted.\n', ...
                fullfile(experimentdir{s},'meshes') );
            s = [];
        end
    end
end
