function viewVolume(varargin)
% function viewVolume(varargin)
%
% Description:
% Utility function that enables users to view a volume given the path
% and name of a volume directory containing the image slices in any image format 
% known to Matlab. If no volume directory is given users are presented
% with an open file dialog to choose a volume slice. All images with the
% same extension in the same directory are assumed to compose the volume. 
% Specifying path in this case causes the file dialog to open in that
% location which. Keeping the volume directory and the path to it separate
% increases reusability of the function as it makes it useful in the
% context of other programs that maintain the notion of a project.
% Furthermore if both the path and the volume directory are specified, a
% third parameter stipulating the image file type has to be present. For
% instance to invoke the viewer without presenting a file dialog use the
% following function call syntax:

% viewVolume('path', fullfile(pwd, 'demoData'), 'volumedirectory', ..
% '672clippedA128Cubed', 'extension', '.png')

% This will display a volume whose slices are located in
% C:\myVolumes\sampleVolumeSubDir'
% 
% Author: Johann Strasser
% Date: 070312

error(nargchk(0, inf, nargin));

if mod(length(varargin), 2) ~= 0
    % input did not come in parameter/value pairs
    error('Arguments have to come in parameter/value pairs.');
end

path = '';
volumeDirectory = '';
extension = '';
fullVolumePath = '';

% For now we ignore any input arguments
for i=1:2:length(varargin)
    switch lower(varargin{i})
        case 'path'
            path = varargin{i + 1};
        case 'volumedirectory'
            volumeDirectory = varargin{i + 1};
        case 'extension'
            extension = varargin{i + 1};
        otherwise
            error(['Unknown parameter name passed to viewVolume. ', ...
                'Parameter name: ',  varargin{i}]);
    end
end

userCancel = 0;

if isequal(volumeDirectory, '')
    
    imreadSupportedFileTypes = getImreadSupportedTypesFilterSpec();    
    [fileName, path] = uigetfile(imreadSupportedFileTypes, ...
        'Choose volume slice', [path, filesep]); 
    
    if isequal(fileName, 0)
        disp('User selected cancel.'); 
        userCancel = 1;
    else
        [fullVolumePath, name, extension, versn] = fileparts(fullfile(path, fileName));       
    end
else
    
    % The extension has to be present for the function dir to be able to
    % list image files of a particular type.
    if ~isequal(extension, '')
        fullVolumePath = [path, filesep, volumeDirectory];
    else
        error(['Image file type extension not specified. ']);
    end
    
end

if ~userCancel
    
    % The following drawnow flushes the event queue and updates the figure
    % window
%     drawnow;
    volume = loadVolumeFromSlicesDirTyped(fullVolumePath, extension);

     % Construct a transform for the volume
     t = getHTTransform();
     t.name = sprintf('transform1');
             
     v = getHTVolume();
     v.name = 'volume1';
     v.parentTransform = 'transform1';
     v.volume = volume;
     v.planes = 256;
     v.voxelSize = [0.001, 0.001, 0.001];
     v.hapticChannel = 'green'; 
     
%      drawnow;
     [transformsOut, volumesOut, pointSetsOut] = hapticTool(t, v, []);
end